package com.aotter.net.controller.eids

import com.aotter.net.dto.eids.request.EidsGenerationRequestBody
import com.aotter.net.dto.eids.request.EidsRefreshRequestBody
import com.aotter.net.dto.eids.response.UserEids
import com.aotter.net.model.repository.eids.EidsRepository
import com.aotter.net.network.Resource
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch

class EidsController(private val eidsRepository: EidsRepository) {

    private val TAG: String = EidsController::class.java.simpleName

    interface OnGenerateListener {
        fun onGenerate(resource: Resource<ArrayList<UserEids>>)
    }

    interface OnRefreshListener {
        fun onRefresh(resource: Resource<UserEids>)
    }

    private val scope: CoroutineScope by lazy {
        CoroutineScope(Dispatchers.Main + SupervisorJob())
    }

    fun generateEids(request: EidsGenerationRequestBody, listener: OnGenerateListener?) =
        scope.launch {
            println("$TAG: generateEids: Starting EIDS generation request")
            println("$TAG: generateEids: Request body - $request")

            eidsRepository.generate(request).onEach { resources ->
                println("$TAG: generateEids: Received response - ${resources.javaClass.simpleName}")
                when (resources) {
                    is Resource.Success -> {
                        println("$TAG: generateEids: Success - Generated ${resources.data?.size ?: 0} EIDS")
                    }

                    is Resource.Error -> {
                        println("$TAG: generateEids: Error - ${resources.errorMessage}")
                    }
                }

                listener?.onGenerate(resources)
                println("$TAG: generateEids: Listener callback executed")

            }.launchIn(this)

        }

    fun refreshEids(request: EidsRefreshRequestBody, listener: OnRefreshListener?) = scope.launch {
        println("$TAG: refreshEids: Starting EIDS refresh request")
        println("$TAG: refreshEids: Request body - $request")

        eidsRepository.refresh(request).onEach { resources ->
            println("$TAG: refreshEids: Received response - ${resources.javaClass.simpleName}")
            when (resources) {
                is Resource.Success -> {
                    println("$TAG: refreshEids: Success - Refreshed EIDS: ${resources.data}")
                }

                is Resource.Error -> {
                    println("$TAG: refreshEids: Error - ${resources.errorMessage}")
                }
            }

            listener?.onRefresh(resources)
            println("$TAG: refreshEids: Listener callback executed")

        }.launchIn(this)

    }

}