package com.aotter.net.dto.trek.response


import android.content.Context
import android.graphics.drawable.Drawable
import android.net.Uri
import android.util.Base64
import android.util.Log
import android.view.View
import com.aotter.net.dto.OmVerificationScriptResource
import com.aotter.net.dto.tracker.response.Error
import com.aotter.net.trek.ads.CustomAdListener
import com.aotter.net.trek.ads.TrekAdListener
import com.aotter.net.trek.ads.controller.trek.TrekAdController
import com.aotter.net.trek.om.TrekOmFactory
import com.aotter.net.trek.sealed.SrcShowField
import com.aotter.net.utils.TrekSdkSettingsUtils
import com.aotter.trek.impression.ImpressionListener
import kotlinx.serialization.Contextual
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

@Serializable
data class TrekNativeAdDto(
    val success: TrekNativeAd? = null,
    val errors: List<Error>? = null
)

@Serializable
data class TrekNativeAd(
    val adType: String,
    val allowCustomClickHandler: Boolean,
    val advertiserName: String,
    val callToAction: String? = null,
    val category: String,
    val sponsor: String,
    @SerialName("imgs") val images: List<ImgSrc>,
    @SerialName("text") val body: String,
    @SerialName("title") val headline: String,
    var imgMain: TrekNativeAdImage = TrekNativeAdImage(),
    var imgIcon: TrekNativeAdImage = TrekNativeAdImage(),
    var imgIconHd: TrekNativeAdImage = TrekNativeAdImage(),
    val unitInstanceId: String,
    var mediaContentAspectRatio: Float? = null,
    @SerialName("url_clc") internal val clickUrl: String,
    @SerialName("url_imp") internal val impressionUrl: String,
    @SerialName("url_original") internal val originalUrl: String,
    @SerialName("third_party_clc") internal val thirdPartyClickUrls: List<String>,
    @SerialName("third_party_imp") internal val thirdPartyImpUrls: List<String>,
    @SerialName("suprAdSrc") internal val mediaSrc: MediaSrc,
    @SerialName("impSetting") internal val impressionSetting: ImpressionSetting,
    val uuid: String,
    internal val validUntil: Long? = null,
    internal val omRes: List<OmVerificationScriptResource>? = listOf(),
    internal val setId: String,
    internal var contentUrl: String = "",
    internal var contentTitle: String = "",
    internal var trekAdListener: TrekAdListener? = null,
    internal var impressionListener: ImpressionListener? = null,
    @Contextual internal var trekAdController: TrekAdController? = null,
    @Contextual internal var trekOmFactory: TrekOmFactory? = null
) {

    fun isExpired(): Boolean {

        return validUntil?.let { validUntil ->

            val currentTimeMillis = System.currentTimeMillis()

            validUntil <= currentTimeMillis

        } ?: kotlin.run {

            false

        }

    }

    fun isVideoAd(): Boolean {

        return (mediaSrc.srcShowField == SrcShowField.VAST_URL.type) || (mediaSrc.srcShowField == SrcShowField.VAST_XML.type) || (mediaSrc.vastTag != null && mediaSrc.vastTag.isNotEmpty()) || (mediaSrc.vastXml != null && mediaSrc.vastXml.isNotEmpty())

    }

    fun setNativeAdClickAction(view: View?) {

        view?.setOnClickListener {

            executeAdClickLogic(it.context)

        }

    }

    fun executeAdClickLogic(context: Context) {

        try {

            val trekAdApiModel = trekAdController ?: return

            val trekAdListener = trekAdListener

            trekAdApiModel.getClickEvent(clickUrl)

            thirdPartyClickUrls.forEach {
                trekAdApiModel.getThirdPartyClickEvent(it)
            }

            trekAdListener?.onAdClicked()

            if (allowCustomClickHandler) {
                Log.d(
                    "TrekNativeAd",
                    "Custom click handler allowed, trekAdListener: \$trekAdListener"
                )
                when (trekAdListener) {
                    is CustomAdListener -> {
                        Log.d(
                            "TrekNativeAd",
                            "CustomAdListener detected, calling onCustomClicked with url: \$originalUrl"
                        )
                        trekAdListener.onCustomClicked(originalUrl)
                    }

                    else -> {
                        Log.d(
                            "TrekNativeAd",
                            "Default handler, opening browser with url: \$originalUrl"
                        )
                        TrekSdkSettingsUtils.openBrowser(context, originalUrl)
                    }
                }
            } else {
                Log.d(
                    "TrekNativeAd",
                    "Custom click handler not allowed, opening browser with url: \$originalUrl"
                )
                TrekSdkSettingsUtils.openBrowser(context, originalUrl)
            }
        } catch (e: Exception) {

            Log.e("nativeAdClickAction", e.toString())

        }

    }

}


@Serializable
data class ImgSrc(
    val height: Int,
    val label: String,
    val src: String,
    val width: Int,
    var image: TrekNativeAdImage = TrekNativeAdImage(),
)

@Serializable
data class ImpressionSetting(
    val impDetectPercent: Double, val impRefreshMillis: Long
)

@Serializable
data class MediaSrc(
    @SerialName("bg_placeholder") val backgroundPlaceHolder: String,
    val height: Int,
    val width: Int,
    val vastTag: String? = null,
    val vastXml: String? = null,
    @SerialName("widget_url")val widgetUrl: String? = null,
    @SerialName("widget_html")var widgetHtml: String? = null,
    val adm: String? = null,
    val srcShowField: String? = null
) {

    fun getAdTagUri(): Uri? {

        return when (srcShowField) {
            SrcShowField.VAST_URL.type -> {

                vastTag?.let { url ->
                    Uri.parse(url)
                } ?: kotlin.run {
                    null
                }

            }


            SrcShowField.VAST_XML.type -> {

                vastXml?.let { xml ->
                    Uri.parse(
                        "data:" + "text/xml" + ";base64," + Base64.encodeToString(
                            xml.toByteArray(), Base64.NO_WRAP
                        )
                    )
                } ?: kotlin.run {
                    null
                }

            }
            else -> vastTag?.let { url ->
                Uri.parse(url)
            } ?: kotlin.run {
                null
            }
        }

    }

}

@Serializable
data class TrekNativeAdImage(
    @Contextual val uri: Uri? = null,
    @Contextual val drawable: Drawable? = null,
)


