package com.aotter.net.model.repository.eids

import com.aotter.net.dto.eids.request.EidsGenerationRequestBody
import com.aotter.net.dto.eids.request.EidsRefreshRequestBody
import com.aotter.net.dto.eids.response.UserEids
import com.aotter.net.dto.trek.response.Errors
import com.aotter.net.network.Resource
import com.aotter.net.service.eids.EidsService
import com.aotter.net.utils.TrekSdkSettingsUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.serialization.decodeFromString
import retrofit2.HttpException
import java.io.IOException
import java.util.concurrent.atomic.AtomicLong

class EidsRepository(
    private val eidsService: EidsService
) {

    private val TAG: String = EidsRepository::class.java.simpleName

    // Rate limiting: 2-second cooldown between API calls
    // Using AtomicLong with built-in atomic operations for thread-safe check-and-set
    private val lastGenerateCallTime = AtomicLong(0L)
    private val lastRefreshCallTime = AtomicLong(0L)
    private val cooldownPeriodMs: Long = 2000L

    /**
     * Helper method to check and enforce rate limiting for API calls.
     * Uses AtomicLong's compareAndSet for atomic check-and-update operations.
     * 
     * @param lastCallTime AtomicLong holding the timestamp of the last call
     * @param methodName Name of the method for error message
     * @return null if rate limit is not exceeded, or error message if it is
     */
    private fun checkRateLimit(lastCallTime: AtomicLong, methodName: String): String? {
        val currentTime = System.currentTimeMillis()
        val lastTime = lastCallTime.get()
        
        // If not enough time has passed, return rate limit error
        if (currentTime - lastTime < cooldownPeriodMs) {
            return "Rate limit exceeded. Please wait before making another $methodName request."
        }
        
        // Use compareAndSet to atomically update the timestamp if it hasn't changed
        // This ensures thread safety without synchronized blocks
        return if (lastCallTime.compareAndSet(lastTime, currentTime)) {
            null // Success - no rate limit exceeded
        } else {
            // Another thread updated the timestamp, retry the check
            checkRateLimit(lastCallTime, methodName)
        }
    }

    suspend fun generate(request: EidsGenerationRequestBody): Flow<Resource<ArrayList<UserEids>>> =
        flow<Resource<ArrayList<UserEids>>> {

            // Check rate limiting - enforce 2-second cooldown
            val rateLimitError = checkRateLimit(lastGenerateCallTime, "generate")
            if (rateLimitError != null) {
                emit(Resource.Error(errorMessage = rateLimitError))
                return@flow
            }

            try {

                val eidsGenerateResponse =
                    if (TrekSdkSettingsUtils.ATSSPEIDsAPIDomain.isNotEmpty() && TrekSdkSettingsUtils.ATSSPPathEIDsAPIGenerate.isNotEmpty()) {
                        val url =
                            "${TrekSdkSettingsUtils.ATSSPEIDsAPIDomain}${TrekSdkSettingsUtils.ATSSPPathEIDsAPIGenerate}"
                        eidsService.generate(url, request)
                    } else {
                        eidsService.generate(request)
                    }

                emit(Resource.Success(eidsGenerateResponse))

            } catch (e: HttpException) {

                val errorMessage = e.response()?.errorBody()?.let {
                    TrekSdkSettingsUtils.json.decodeFromString<Errors>(it.string()).errors[0].message
                } ?: kotlin.run {
                    "An unexpected error occurred"
                }

                emit(Resource.Error(errorMessage = errorMessage))

            } catch (e: IOException) {

                emit(Resource.Error(errorMessage = "Couldn't reach server. Check your internet connection."))

            }

        }.catch {

            println("$TAG: ${it.localizedMessage ?: "request ad error."}")

            emit(Resource.Error(errorMessage = "An unexpected error occurred."))

        }.flowOn(Dispatchers.IO)


    suspend fun refresh(request: EidsRefreshRequestBody): Flow<Resource<UserEids>> =
        flow<Resource<UserEids>> {

            // Check rate limiting - enforce 2-second cooldown  
            val rateLimitError = checkRateLimit(lastRefreshCallTime, "refresh")
            if (rateLimitError != null) {
                emit(Resource.Error(errorMessage = rateLimitError))
                return@flow
            }

            try {

                val eidsRefreshResponse =
                    if (TrekSdkSettingsUtils.ATSSPEIDsAPIDomain.isNotEmpty() && TrekSdkSettingsUtils.ATSSPPathEIDsAPIRefresh.isNotEmpty()) {
                        val url =
                            "${TrekSdkSettingsUtils.ATSSPEIDsAPIDomain}${TrekSdkSettingsUtils.ATSSPPathEIDsAPIRefresh}"
                        eidsService.refresh(url, request)
                    } else {
                        eidsService.refresh(request)
                    }

                emit(Resource.Success(eidsRefreshResponse))

            } catch (e: HttpException) {

                val errorMessage = e.response()?.errorBody()?.let {
                    TrekSdkSettingsUtils.json.decodeFromString<Errors>(it.string()).errors[0].message
                } ?: kotlin.run {
                    "An unexpected error occurred"
                }

                emit(Resource.Error(errorMessage = errorMessage))

            } catch (e: IOException) {

                emit(Resource.Error(errorMessage = "Couldn't reach server. Check your internet connection."))

            }

        }.catch {

            println("$TAG: ${it.localizedMessage ?: "request ad error."}")

            emit(Resource.Error(errorMessage = "An unexpected error occurred."))

        }.flowOn(Dispatchers.IO)

}