package com.aotter.net.model.repository.trek

import android.graphics.BitmapFactory
import android.graphics.drawable.Drawable
import android.net.Uri
import android.util.Log
import com.aotter.net.dto.error.request.ErrorReport
import com.aotter.net.dto.trek.request.AdBo
import com.aotter.net.dto.trek.response.Errors
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.dto.trek.response.TrekNativeAdDto
import com.aotter.net.dto.trek.response.TrekNativeAdImage
import com.aotter.net.network.Resource
import com.aotter.net.service.trek.TrekService
import com.aotter.net.trek.TrekAds
import com.aotter.net.trek.TrekDataKey
import com.aotter.net.trek.ads.NativeAdOptions
import com.aotter.net.utils.ErrorReportUtils
import com.aotter.net.utils.TrekSdkSettingsUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.withContext
import kotlinx.serialization.decodeFromString
import okhttp3.ResponseBody
import retrofit2.HttpException
import java.io.IOException
import java.net.URL


class TrekRepository(private val trekService: TrekService, private val commonService: TrekService) {

    private var nativeAdOptions: NativeAdOptions? = null

    fun setNativeAdOptions(options: NativeAdOptions?) {
        this.nativeAdOptions = options
    }

    private val TAG: String = TrekRepository::class.java.simpleName

    fun init(adBo: AdBo): Flow<ResponseBody> = flow<ResponseBody> {
        try {
            trekService.init(adBo)
        } catch (e: HttpException) {
            Log.i(TAG, e.localizedMessage ?: "Sent init event error.")
        } catch (e: IOException) {
            Log.i(TAG, e.localizedMessage ?: "Sent init event error.")
        }
    }

    suspend fun postAd(adBo: AdBo): Flow<Resource<TrekNativeAd>> = flow<Resource<TrekNativeAd>> {

        try {
            val trekNativeAd =
                if (TrekSdkSettingsUtils.ATSSPDomain.isNotEmpty() && TrekSdkSettingsUtils.ATSSPPathFetchAd.isNotEmpty()) {
                    val url =
                        "${TrekSdkSettingsUtils.ATSSPDomain}${TrekSdkSettingsUtils.ATSSPPathFetchAd}"
                    trekService.postAd(url, adBo)
                } else {
                    trekService.postAd(adBo)
                }

            trekNativeAd.success?.apply {

                val nativeAdData = setDataToTrekNativeAd(adBo, this@apply)

                emit(Resource.Success(nativeAdData))

            } ?: kotlin.run {

                val errorMessage =
                    trekNativeAd.errors?.get(0)?.message ?: "An unexpected error occurred"

                emit(Resource.Error(errorMessage = errorMessage))

            }

        } catch (e: HttpException) {

            val errorMessage = e.response()?.errorBody()?.let {
                TrekSdkSettingsUtils.json.decodeFromString<Errors>(it.string()).errors[0].message
            } ?: kotlin.run {
                "An unexpected error occurred"
            }

            emit(Resource.Error(errorMessage = errorMessage))

        } catch (e: IOException) {

            emit(Resource.Error(errorMessage = "Couldn't reach server. Check your internet connection."))

        }

    }.catch {

        Log.i(TAG, it.localizedMessage ?: "request ad error.")

        val errorReport = ErrorReport<AdBo, TrekNativeAdDto>(
            adBo,
            null,
            it.stackTraceToString()
        )

        ErrorReportUtils.sendFetchAdError(errorReport)

        emit(Resource.Error(errorMessage = "An unexpected error occurred."))

    }.flowOn(Dispatchers.IO)

    suspend fun getClickEvent(url: String): Flow<ResponseBody> = flow<ResponseBody> {

        try {

            val data = commonService.getClickEvent(url)

            data.body()?.let {

                Log.i(TAG, "Sent click event success.")

            } ?: kotlin.run {

                Log.i(TAG, "Sent click event error.")

            }

        } catch (e: HttpException) {

            Log.i(TAG, e.localizedMessage ?: "Sent click event error.")

        } catch (e: IOException) {

            Log.i(TAG, e.localizedMessage ?: "Sent click event error.")

        }

    }.catch {

        Log.i(TAG, it.localizedMessage ?: "An unexpected error occurred.")

    }.flowOn(Dispatchers.IO)

    suspend fun getImpressionEvent(url: String): Flow<ResponseBody> = flow<ResponseBody> {

        try {

            val data = commonService.getImpressionEvent(url)

            data.body()?.let {

                Log.i(TAG, "Sent impression event success.")

            } ?: kotlin.run {

                Log.i(TAG, "Sent impression event error.")

            }

        } catch (e: HttpException) {

            Log.i(TAG, e.localizedMessage ?: "Sent impression event error.")

        } catch (e: IOException) {

            Log.i(TAG, e.localizedMessage ?: "Sent impression event error.")

        }

    }.catch {

        Log.i(TAG, it.localizedMessage ?: "An unexpected error occurred.")

    }.flowOn(Dispatchers.IO)

    suspend fun getThirdPartyClickEvent(url: String): Flow<ResponseBody> = flow<ResponseBody> {

        try {

            val data = commonService.getThirdPartyClickEvent(url)


            data.body()?.let {

                Log.i(TAG, "Sent third party click event success.")

            } ?: kotlin.run {

                Log.i(TAG, "Sent third party click event error.")

            }

        } catch (e: HttpException) {

            Log.i(TAG, e.localizedMessage ?: "Sent third party click event error.")

        } catch (e: IOException) {

            Log.i(TAG, e.localizedMessage ?: "Sent third party click event error.")

        }

    }.catch {

        Log.i(TAG, it.localizedMessage ?: "An unexpected error occurred.")

    }.flowOn(Dispatchers.IO)

    suspend fun getThirdPartyImpressionEvent(url: String): Flow<ResponseBody> = flow<ResponseBody> {

        try {

            val data = commonService.getThirdPartyImpressionEvent(url)

            data.body()?.let {

                Log.i(TAG, "Sent third party impression event success.")

            } ?: kotlin.run {

                Log.i(TAG, "Sent third party impression event error.")

            }

        } catch (e: HttpException) {

            Log.i(TAG, e.localizedMessage ?: "Sent third party impression event error.")

        } catch (e: IOException) {

            Log.i(TAG, e.localizedMessage ?: "Sent third party impression event error.")

        }

    }.catch {

        Log.i(TAG, it.localizedMessage ?: "An unexpected error occurred.")

    }.flowOn(Dispatchers.IO)

    private suspend fun downloadImage(url: String?, isBackgroundImg: Boolean): Drawable? =
        withContext(Dispatchers.IO) {
            if (!isBackgroundImg && nativeAdOptions?.shouldReturnUrlsForImageAssets() == true) {
            return@withContext null
        }

        try {
            if (url.isNullOrEmpty() || !url.startsWith("http")) {
                return@withContext null
            }

            val context = TrekAds.getApplicationContext() ?: return@withContext null

            TrekSdkSettingsUtils.getImageCache()[url]?.let { bitmap ->
                return@withContext TrekSdkSettingsUtils.getBitmapDrawable(context, bitmap)
            }

            val bitmap = URL(url).openConnection().apply {
                connectTimeout = 10000
                readTimeout = 15000
                useCaches = true
            }.inputStream.buffered().use { input ->
                BitmapFactory.decodeStream(input)
            } ?: return@withContext null

            TrekSdkSettingsUtils.setImageCache(url, bitmap)

            return@withContext TrekSdkSettingsUtils.getBitmapDrawable(context, bitmap)

        } catch (e: Exception) {
            return@withContext null
        }
    }

    private suspend fun downloadHtmlContentFromUrl(urlString: String): String? =
        withContext(Dispatchers.IO) {

            return@withContext try {

                val url = URL(urlString)

                url.openStream().use { stream ->

                    val htmlContent = stream.bufferedReader().use { it.readText() }

                    htmlContent

                }

            } catch (e: Exception) {

                null

            }

        }

    private fun getAspectRatio(trekNativeAd: TrekNativeAd): Float? {

        if (trekNativeAd.adType == TrekDataKey.BANNER) {

            return null

        }

        return if (trekNativeAd.isVideoAd()) {

            .5625f

        } else {

            val scale =
                TrekAds.getApplicationContext()?.resources?.displayMetrics?.density ?: return .5625f

            val w = if (trekNativeAd.mediaSrc.width == 0) {
                (1200 * scale + 0.5f).toInt()
            } else {
                (trekNativeAd.mediaSrc.width * scale + 0.5f).toInt()
            }

            val h = if (trekNativeAd.mediaSrc.height == 0) {
                (628 * scale + 0.5f).toInt()
            } else {
                (trekNativeAd.mediaSrc.height * scale + 0.5f).toInt()
            }

            return (h.toFloat() / w.toFloat())

        }

    }

    private suspend fun setDataToTrekNativeAd(
        adBo: AdBo, trekNativeAd: TrekNativeAd
    ): TrekNativeAd = withContext(Dispatchers.IO) {

        trekNativeAd.contentUrl = adBo.payload.contentUrl

        trekNativeAd.contentTitle = adBo.payload.contentTitle

        trekNativeAd.mediaContentAspectRatio = getAspectRatio(trekNativeAd)

        if (trekNativeAd.mediaSrc.backgroundPlaceHolder.isNotEmpty() && trekNativeAd.isVideoAd()) {

            downloadImage(
                trekNativeAd.mediaSrc.backgroundPlaceHolder,
                true
            )

        }

        trekNativeAd.images.filter { it.src.isNotEmpty() }.forEach { img ->

            val drawable = downloadImage(img.src, false)

            val trekNativeAdImage = TrekNativeAdImage(
                Uri.parse(img.src), drawable
            )

            img.image = trekNativeAdImage

            when (img.label) {

                "img_main" -> trekNativeAd.imgMain = trekNativeAdImage

                "img_icon" -> trekNativeAd.imgIcon = trekNativeAdImage

                "img_icon_hd" -> trekNativeAd.imgIconHd = trekNativeAdImage

            }

        }

        return@withContext trekNativeAd

    }


}