package com.aotter.net.network

import com.aotter.net.BuildConfig
import com.aotter.net.utils.TrekSdkSettingsUtils
import com.jakewharton.retrofit2.converter.kotlinx.serialization.asConverterFactory
import okhttp3.Interceptor
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import java.io.IOException
import java.util.concurrent.TimeUnit


object RetrofitBuilder {
    val loggingInterceptor = HttpLoggingInterceptor().apply {
        level = HttpLoggingInterceptor.Level.BODY // Set the desired logging level
    }

    fun createOkHttpClient(interceptor: HttpLoggingInterceptor? = null): OkHttpClient {

        val httpClient = OkHttpClient.Builder()
        interceptor?.let { httpClient.addInterceptor(it) }
        val httpLoggingInterceptor = HttpLoggingInterceptor()

//        httpLoggingInterceptor.level = if(BuildConfig.DEBUG){
//
//            HttpLoggingInterceptor.Level.BODY
//
//        }else{
//
//            HttpLoggingInterceptor.Level.NONE
//
//        }

        httpClient
            .retryOnConnectionFailure(true) // Default retry once, implement interceptor if need to retry n times
            .connectTimeout(20, TimeUnit.SECONDS)
            .readTimeout(20, TimeUnit.SECONDS)
            .followRedirects(false)
            .followSslRedirects(false)
            .addInterceptor(httpLoggingInterceptor)

        httpClient.addInterceptor(RedirectInterceptor())


        return httpClient.build()
    }

    fun createEidsOkHttpClient(interceptor: HttpLoggingInterceptor? = null): OkHttpClient {

        val httpClient = OkHttpClient.Builder()
        interceptor?.let { httpClient.addInterceptor(it) }
        val httpLoggingInterceptor = HttpLoggingInterceptor()

        httpClient
            .retryOnConnectionFailure(true) // Default retry once, implement interceptor if need to retry n times
            .connectTimeout(2, TimeUnit.SECONDS)
            .readTimeout(2, TimeUnit.SECONDS)
            .followRedirects(false)
            .followSslRedirects(false)
            .addInterceptor(httpLoggingInterceptor)

        httpClient.addInterceptor(RedirectInterceptor())

        return httpClient.build()
    }

    inline fun <reified T> createCommonService(): T {

        val contentType = "application/json".toMediaType()

        val retrofit = Retrofit.Builder()
            .baseUrl(BuildConfig.MFTC_URL)
            .addConverterFactory(TrekSdkSettingsUtils.json.asConverterFactory(contentType))
            .build()
        return retrofit.create(T::class.java)
    }

    inline fun <reified T> createTrekService(url: String = BuildConfig.MFTC_URL): T {

        val contentType = "application/json".toMediaType()

        val retrofit = Retrofit.Builder()
            .baseUrl(url)
            .client(createOkHttpClient(loggingInterceptor))
            .addConverterFactory(TrekSdkSettingsUtils.json.asConverterFactory(contentType))
            .build()
        return retrofit.create(T::class.java)
    }

    inline fun <reified T> createTrekErrorService(): T {

        val contentType = "application/json".toMediaType()

        val retrofit = Retrofit.Builder()
            .baseUrl(BuildConfig.ERROR_REPORT_URL)
            .client(createOkHttpClient(loggingInterceptor))
            .addConverterFactory(TrekSdkSettingsUtils.json.asConverterFactory(contentType))
            .build()
        return retrofit.create(T::class.java)
    }

    inline fun <reified T> createEidsService(): T {

        val contentType = "application/json".toMediaType()

        val retrofit = Retrofit.Builder()
            .baseUrl(BuildConfig.EIDS_URL)
            .client(createEidsOkHttpClient(loggingInterceptor))
            .addConverterFactory(TrekSdkSettingsUtils.json.asConverterFactory(contentType))
            .build()
        return retrofit.create(T::class.java)
    }

    inline fun <reified T> createTrackerService(): T {

        val contentType = "application/json".toMediaType()

        val retrofit = Retrofit.Builder()
            .baseUrl(BuildConfig.TRACKER_URL)
            .client(createOkHttpClient(loggingInterceptor))
            .addConverterFactory(TrekSdkSettingsUtils.json.asConverterFactory(contentType))
            .build()
        return retrofit.create(T::class.java)
    }

    private fun createResponse(chain: Interceptor.Chain, request: Request): Response {

        return chain.proceed(
            request.newBuilder()
                .addHeader(
                    "x-aotter-clientid",
                    TrekSdkSettingsUtils.getClientId()
                )
                .addHeader(
                    "x-aotter-version",
                    "${TrekSdkSettingsUtils.OS}_${TrekSdkSettingsUtils.SDK_VERSION}"
                )
                .addHeader("Content-Type", "application/json")
                .removeHeader("User-Agent") // Remove the old one
                .addHeader("User-Agent", TrekSdkSettingsUtils.getDefaultUserAgent())
                .method(request.method, request.body).build()
        )

    }


    class RedirectInterceptor : Interceptor {

        @Throws(IOException::class)
        override fun intercept(chain: Interceptor.Chain): Response {

            val request = chain.request()

            val response = createResponse(chain, request)

            when (response.code) {
                in 500..900 -> {

                    throw IOException(response.body?.string())

                }
            }

            return response

        }
    }

}



