package com.aotter.net.trek

import android.content.Context
import android.os.Build
import android.util.Log
import androidx.annotation.OptIn
import androidx.media3.common.MediaLibraryInfo
import androidx.media3.common.util.UnstableApi
import com.aotter.net.BuildConfig
import com.aotter.net.dto.TrekJsonObject
import com.aotter.net.dto.trek.request.AdBo
import com.aotter.net.dto.trek.request.Payload
import com.aotter.net.model.repository.trek.TrekRepository
import com.aotter.net.network.RetrofitBuilder
import com.aotter.net.trek.ads.controller.trek.TrekAdController
import com.aotter.net.trek.om.TrekOmLauncher
import com.aotter.net.utils.AppInfoUtils
import com.aotter.net.utils.DeviceUtils
import com.aotter.net.utils.EidsUtils
import com.aotter.net.utils.TrekSdkSettingsUtils
import com.aotter.net.utils.UserInfoUtils
import com.iab.omid.library.aotternet.Omid
import java.util.Locale

object TrekAds {

    private val TAG: String = TrekAds::class.java.simpleName

    private var applicationContext: Context? = null
    internal lateinit var trekAdController: TrekAdController

    init {
        initTrekController()
    }

    private fun initTrekController() {

        this.trekAdController = TrekAdController(
            trekRepository = TrekRepository(
                RetrofitBuilder.createTrekService(BuildConfig.MFTC_URL),
                RetrofitBuilder.createCommonService()
            )
        )

    }

    interface OnInitializationCompleteListener {

        fun onInitializationComplete()

    }

    /**
     * Returns your app's applicationContext.
     */
    fun getApplicationContext() = applicationContext

    /**
     * Returns SDK version.
     */
    fun getSdkVersion() = BuildConfig.SDK_VERSION

    @JvmStatic
    fun initialize(
        context: Context,
        clientId: String,
        OnInitializationCompleteListener: OnInitializationCompleteListener
    ) = initialize(
        context = context,
        clientId = clientId,
        OnInitializationCompleteListener = OnInitializationCompleteListener,
        ATSSPDomain = "",
        ATSSPPathFetchAd = "",
        ATSSPEIDsAPIDomain = "",
        ATSSPPathEIDsAPIGenerate = "",
        ATSSPPathEIDsAPIRefresh = "",
        ATTrackerDomain = "",
        ATTrackerPathReport = "",
    )

    /**
     * The initialize() method must be called before requests ad at least once.
     * @param context
     * Injecting context.
     * @param clientId
     * Injecting Trek client id.
     * @param OnInitializationCompleteListener
     * It is a interface which notify developer the  initialize() method is complete or not.
     */
    fun initialize(
        context: Context,
        clientId: String,
        OnInitializationCompleteListener: OnInitializationCompleteListener,
        ATSSPDomain: String = "",
        ATSSPPathFetchAd: String = "",
        ATSSPEIDsAPIDomain: String = "",
        ATSSPPathEIDsAPIGenerate: String = "",
        ATSSPPathEIDsAPIRefresh: String = "",
        ATTrackerDomain: String = "",
        ATTrackerPathReport: String = "",
    ) {

        try {

            if (TrekSdkSettingsUtils.getClientId().isNotEmpty()) {

                TrekSdkSettingsUtils.setClientId(clientId)

                OnInitializationCompleteListener.onInitializationComplete()

                return

            }

            applicationContext = context.applicationContext.apply {


                TrekSdkSettingsUtils.updateSettings(
                    ATSSPDomain = ATSSPDomain,
                    ATSSPPathFetchAd = ATSSPPathFetchAd,
                    ATSSPEIDsAPIDomain = ATSSPEIDsAPIDomain,
                    ATSSPPathEIDsAPIGenerate = ATSSPPathEIDsAPIGenerate,
                    ATSSPPathEIDsAPIRefresh = ATSSPPathEIDsAPIRefresh,
                    ATTrackerDomain = ATTrackerDomain,
                    ATTrackerPathReport = ATTrackerPathReport
                )

                TrekSdkSettingsUtils.setClientId(clientId)

                TrekOmLauncher.activateOm(this)

                DeviceUtils.setDevice(this)

                TrekSdkSettingsUtils.initImageCache(this)

                TrekSdkSettingsUtils.initUnitInstanceIdCache(this)

                TrekSdkSettingsUtils.setDefaultUserAgent(this)

                UserInfoUtils.init(this)

                AppInfoUtils.init(this)

                EidsUtils.init(this)


                val adBo = AdBo(
                    DeviceUtils.getDevice(),
                    UserInfoUtils.getUserInfo(),
                    "${TrekSdkSettingsUtils.OS}_${TrekSdkSettingsUtils.SDK_VERSION}",
                    TrekSdkSettingsUtils.SDK_VERSION_CODE.toInt(),
                    "",
                    0,
                    "",
                    Payload(
                        "",
                        "",
                        "",
                        TrekJsonObject()
                    ),
                    TrekSdkSettingsUtils.ATSSPDomain,
                    TrekSdkSettingsUtils.ATSSPPathFetchAd,
                    TrekSdkSettingsUtils.ATSSPEIDsAPIDomain,
                    TrekSdkSettingsUtils.ATSSPPathEIDsAPIGenerate,
                    TrekSdkSettingsUtils.ATSSPPathEIDsAPIRefresh,
                    TrekSdkSettingsUtils.ATTrackerDomain,
                    TrekSdkSettingsUtils.ATTrackerPathReport
                )
                trekAdController.init(adBo)

                OnInitializationCompleteListener.onInitializationComplete()

                printInitializedLog()

            }

        } catch (e: Exception) {

            throw Exception(e.toString())

        }

    }

    @OptIn(UnstableApi::class)
    private fun printInitializedLog() {

        Log.i(
            TAG,
            "[TrekSdk]\n" +
                    "========================================\n" +
                    "=====Trek SDK=====\n" +
                    "===SDK Versions===\n" +
                    "Trek SDK Version: ${BuildConfig.SDK_VERSION}\n" +
                    "Trek SDK Version Code: ${BuildConfig.SDK_VERSION_CODE}\n" +
                    "OM SDK Version: ${Omid.getVersion()}\n" +
                    "ExoPlayer SDK Version: ${MediaLibraryInfo.VERSION}\n" +
                    "===Trek SDK Settings===\n" +
                    "Client ID: ${TrekSdkSettingsUtils.getClientId()}\n" +
                    "===Device Info===\n" +
                    "OS Version: ${DeviceUtils.getDevice().osVersion}\n" +
                    "API Level: ${DeviceUtils.getDevice().SDK_INT}\n" +
                    "Build ID: ${Build.ID}\n" +
                    "GAID: ${DeviceUtils.getDevice().adID}\n" +
                    "Device ID: ${DeviceUtils.getDevice().deviceID}\n" +
                    "Device Brand: ${DeviceUtils.getDevice().deviceBrand}\n" +
                    "Device Model: ${DeviceUtils.getDevice().deviceModel}\n" +
                    "Network Cond: ${DeviceUtils.getDevice().networkCond}\n" +
                    "Locale: ${Locale.getDefault()}\n" +
                    "========================================\n"
        )

    }

}
