package com.aotter.net.trek.ads

import android.content.Context
import androidx.annotation.IntRange
import com.aotter.net.BuildConfig.MFTC_URL
import com.aotter.net.dto.trek.request.AdBo
import com.aotter.net.dto.trek.request.Payload
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.model.repository.trek.TrekRepository
import com.aotter.net.network.Resource
import com.aotter.net.network.RetrofitBuilder
import com.aotter.net.trek.TrekDataKey
import com.aotter.net.trek.ads.controller.trek.TrekAdController
import com.aotter.net.utils.DeviceUtils
import com.aotter.net.utils.TrekSdkSettingsUtils
import com.aotter.net.utils.UserInfoUtils

class TrekAdLoader(private val build: Builder) {

    fun loadAds(
        trekAdRequest: TrekAdRequest,
        @IntRange(from = 2, to = 5) maxNumberOfAds: Int
    ) {

        checkClientId()

        updateUserEids()

        build.trekAdController.postAds(getAdBo(trekAdRequest), maxNumberOfAds)

    }

    fun loadAd(trekAdRequest: TrekAdRequest) {

        checkClientId()

        updateUserEids()

        build.trekAdController.postAd(getAdBo(trekAdRequest))

    }

    private fun getAdBo(trekAdRequest: TrekAdRequest) = AdBo(
        DeviceUtils.getDevice(),
        UserInfoUtils.getUserInfo(),
        "${TrekSdkSettingsUtils.OS}_${TrekSdkSettingsUtils.SDK_VERSION}",
        TrekSdkSettingsUtils.SDK_VERSION_CODE.toInt(),
        trekAdRequest.getMediationVersion(),
        trekAdRequest.getMediationVersionCode(),
        build.placeUid,
        Payload(
            trekAdRequest.getCategory(),
            trekAdRequest.getContentUrl(),
            trekAdRequest.getContentTitle(),
            trekAdRequest.getMeta()
        )
    )

    class Builder(
        private val context: Context,
        placeUid: String
    ) {
        internal var placeUid = placeUid
            get() = field

        internal lateinit var trekAdController: TrekAdController

        private var trekAdListener: TrekAdListener? = null
        private var nativeAdOptions: NativeAdOptions? = null

        private var onTrekAdListener = object : TrekAdController.OnTrekAdListener {

            override fun onTrekAds(resources: List<Resource<TrekNativeAd>>) {

                if (resources.isEmpty()) return

                val trekNativeAds = resources.mapNotNull { it.data }

                val errorMessages = resources.mapNotNull { it.errorMessage }

                if (trekNativeAds.isNotEmpty()) {

                    trekNativeAds.forEach { trekNativeAd ->

                        trekNativeAd.trekAdListener = trekAdListener

                        trekNativeAd.trekAdController = trekAdController

                    }

                    trekAdListener?.onAdsLoaded(trekNativeAds)

                    return

                }

                if (errorMessages.isEmpty()) return

                trekAdListener?.onAdsFailedToLoad(
                    errorMessages
                )

            }

            override fun onTrekAd(resource: Resource<TrekNativeAd>) {

                when (resource) {

                    is Resource.Success -> {

                        val trekNativeAd = resource.data ?: return

                        trekNativeAd.trekAdListener = trekAdListener

                        trekNativeAd.trekAdController = trekAdController

                        trekAdListener?.onAdLoaded(trekNativeAd)

                    }

                    is Resource.Error -> {

                        val errorMessage = resource.errorMessage ?: return

                        trekAdListener?.onAdFailedToLoad(errorMessage)

                    }

                }

            }

        }

        init {

            initTrekController()

            initTrekAdListener()

        }

        private fun initTrekController() {
            this.trekAdController = TrekAdController(
                trekRepository = TrekRepository(
                    RetrofitBuilder.createTrekService(MFTC_URL),
                    RetrofitBuilder.createCommonService()
                )
            )

        }

        private fun initTrekAdListener() {

            trekAdController.setOnTrekAdListener(onTrekAdListener)

        }

        /**
         * 廣告事件的監聽
         * @param trekAdListener
         */
        fun withAdListener(trekAdListener: TrekAdListener?): Builder {
            this.trekAdListener = trekAdListener
            return this
        }


        fun withNativeAdOptions(build: NativeAdOptions): Builder {
            this.nativeAdOptions = build
            return this
        }

        fun build(): TrekAdLoader {
            val loader = TrekAdLoader(this)
            trekAdController.setNativeAdOptions(nativeAdOptions)
            return loader
        }
    }

    private fun checkClientId() {
        if (TrekSdkSettingsUtils.getClientId().isEmpty()) {
            throw IllegalArgumentException(TrekDataKey.INITIALIZED_INCORRECTLY)
        }
    }

    private fun updateUserEids() {
        UserInfoUtils.setUserEidsItemList()
    }

}