package com.aotter.net.trek.ads

import android.content.Context
import android.util.AttributeSet
import android.util.Log
import android.view.View
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.constraintlayout.widget.ConstraintSet
import androidx.core.view.forEach
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import com.aotter.net.dto.trek.request.AdBo
import com.aotter.net.dto.trek.request.Payload
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.extension.addObserverExt
import com.aotter.net.extension.removeObserverExt
import com.aotter.net.model.repository.trek.TrekRepository
import com.aotter.net.network.Resource
import com.aotter.net.network.RetrofitBuilder
import com.aotter.net.trek.TrekDataKey
import com.aotter.net.trek.ads.controller.trek.TrekAdController
import com.aotter.net.trek.sealed.RefreshTime
import com.aotter.net.utils.*
import kotlinx.coroutines.*
import java.util.concurrent.ConcurrentLinkedQueue


class TrekBannerAdView : ConstraintLayout, LifecycleEventObserver,
    View.OnAttachStateChangeListener {

    constructor(context: Context) : super(context) {

    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {

    }

    constructor(context: Context, attrs: AttributeSet?, defStyle: Int) : super(
        context,
        attrs,
        defStyle
    ) {

    }

    private var TAG: String = TrekBannerAdView::class.java.simpleName

    private var placeUid: String = ""

    private var trekAdListener: TrekAdListener? = null

    private var trekAdController = TrekAdController(
        trekRepository = TrekRepository(
            RetrofitBuilder.createTrekService(),
            RetrofitBuilder.createCommonService()
        )
    )

    private var timerScope: CoroutineScope? = null

    private var lifeCycle: Lifecycle? = (context as? FragmentActivity)?.lifecycle

    private var trekAdRequest: TrekAdRequest? = null

    private var concurrentLinkedQueue =
        ConcurrentLinkedQueue<Pair<TrekNativeAd, TrekMediaView>>()

    private var currentTrekNativeAd: TrekNativeAd? = null

    private var viewStateTracker: ViewStateTracker? = null

    var preload = false
        set(value) {
            field = value
        }

    var autoRefresh = false
        set(value) {
            field = value
        }

    var refreshTime: RefreshTime = RefreshTime.REFRESH_TIME_15000_MS
        set(value) {
            field = value
        }

    private var onTrekAdListener = object : TrekAdController.OnTrekAdListener {

        override fun onTrekAd(resource: Resource<TrekNativeAd>) {

            when (resource) {

                is Resource.Success -> {

                    val trekNativeAd = resource.data ?: return

                    createView(trekNativeAd)

                    setView()

                }

                is Resource.Error -> {

                    val errorMessage = resource.errorMessage ?: return

                    trekAdListener?.onAdFailedToLoad(errorMessage)

                }

            }

        }

        override fun onTrekAds(resources: List<Resource<TrekNativeAd>>) {

            if (resources.isEmpty()) return

            val trekNativeAds = resources.mapNotNull { it.data }

            val errorMessages = resources.mapNotNull { it.errorMessage }

            if (trekNativeAds.isNotEmpty()) {

                trekNativeAds.forEach {

                    createView(it)

                }

                setView()

                return

            }

            if (errorMessages.isEmpty()) return

            trekAdListener?.onAdFailedToLoad(errorMessages[0])

        }

    }

    fun setPlaceUid(placeUid: String) {

        this.placeUid = placeUid

    }

    fun setAdListener(trekAdListener: TrekAdListener?) {

        this.trekAdListener = trekAdListener

    }

    fun loadAd(trekAdRequest: TrekAdRequest) {

        if (TrekSdkSettingsUtils.getClientId().isEmpty()) {

            throw IllegalArgumentException(TrekDataKey.INITIALIZED_INCORRECTLY)

        }

        addOnAttachStateChangeListener(this)

        this.lifeCycle?.addObserverExt(this)

        this.trekAdRequest = trekAdRequest

        if (concurrentLinkedQueue.isNotEmpty()) {

            setView()

            return

        }

        trekAdController.apply {

            setOnTrekAdListener(onTrekAdListener)

            if (preload) {
                postAds(getAdBo(trekAdRequest), 2)
            } else {
                postAd(getAdBo(trekAdRequest))
            }

        }

    }

    private fun getAdBo(trekAdRequest: TrekAdRequest) = AdBo(
        DeviceUtils.getDevice(),
        UserInfoUtils.getUserInfo(),
        "${TrekSdkSettingsUtils.OS}_${TrekSdkSettingsUtils.SDK_VERSION}",
        TrekSdkSettingsUtils.SDK_VERSION_CODE.toInt(),
        trekAdRequest.getMediationVersion(),
        trekAdRequest.getMediationVersionCode(),
        placeUid,
        Payload(
            trekAdRequest.getCategory(),
            trekAdRequest.getContentUrl(),
            trekAdRequest.getContentTitle(),
            trekAdRequest.getMeta()
        )
    )

    fun resume() {

        viewStateTracker?.resume()

        launchRefreshTimer(trekAdRequest)

    }

    fun pause() {

        viewStateTracker?.pause()

        cancelTimer()

    }

    fun destroy() {

        cancelTimer()

        currentTrekNativeAd?.let {

            TrekAdViewUtils.destroyAd(it)

            viewStateTracker?.destroy()

            viewStateTracker = null

        }

        concurrentLinkedQueue.forEach { pair ->

            val trekMediaView = pair.second

            trekMediaView.destroy()

        }

        concurrentLinkedQueue.clear()

        this.lifeCycle?.removeObserverExt(this)

        removeOnAttachStateChangeListener(this)

        Log.i(TAG, "Banner Ad destroy.")

    }

    private fun createView(trekNativeAd: TrekNativeAd) {

        trekNativeAd.trekAdListener = trekAdListener

        trekNativeAd.trekAdController = trekAdController

        val trekMediaView = TrekMediaView(context, null)

        trekMediaView.id = View.generateViewId()

        concurrentLinkedQueue.offer(Pair(trekNativeAd, trekMediaView))

    }

    private fun cancelTimer() {

        timerScope?.cancel()

        timerScope = null

    }

    private fun launchRefreshTimer(trekAdRequest: TrekAdRequest?) {

        if (trekAdRequest == null) return

        cancelTimer()

        if (!autoRefresh) return

        timerScope = CoroutineScope(Dispatchers.Main + SupervisorJob()).apply {

            launch {

                delay(refreshTime.ms)

                this@TrekBannerAdView.loadAd(trekAdRequest)

            }

        }

    }

    private fun setView() {

        currentTrekNativeAd?.let {

            TrekAdViewUtils.destroyAd(it)

            viewStateTracker?.destroy()

            viewStateTracker = null

        }

        this@TrekBannerAdView.removeAllViews()

        val pair = concurrentLinkedQueue.poll()

        currentTrekNativeAd = pair?.first

        val trekMediaView = pair?.second

        currentTrekNativeAd?.let { currentTrekNativeAd ->

            trekMediaView?.let { trekMediaView ->

                val lp = LayoutParams(
                    LayoutParams.WRAP_CONTENT,
                    LayoutParams.WRAP_CONTENT
                ).apply {
                    bottomToBottom = ConstraintSet.PARENT_ID
                    endToEnd = ConstraintSet.PARENT_ID
                    startToStart = ConstraintSet.PARENT_ID
                    topToTop = ConstraintSet.PARENT_ID
                }

                this@TrekBannerAdView.addView(trekMediaView, lp)

                startViewTracker(currentTrekNativeAd, trekMediaView)

                trekAdListener?.onAdLoaded(currentTrekNativeAd)

                launchRefreshTimer(trekAdRequest)

            }

        }

    }

    override fun onStateChanged(source: LifecycleOwner, event: Lifecycle.Event) {
        when (event) {
            Lifecycle.Event.ON_RESUME -> resume()
            Lifecycle.Event.ON_PAUSE -> pause()
            Lifecycle.Event.ON_DESTROY -> destroy()
            else -> {
            }
        }
    }

    override fun onViewAttachedToWindow(v: View) {

        Log.i(TAG, "Banner Ad attached to window.")

        resume()

    }

    override fun onViewDetachedFromWindow(v: View) {

        Log.i(TAG, "Banner Ad detached from window.")

        pause()

    }


    private fun startViewTracker(
        currentTrekNativeAd: TrekNativeAd,
        trekMediaView: TrekMediaView
    ) {

        viewStateTracker = TrekAdViewUtils.createViewStateTracker(currentTrekNativeAd).apply {

            this@TrekBannerAdView.forEach { view ->

                this.addFriendlyObstruction(view)

            }

            this.launchViewStateTracker(this@TrekBannerAdView, trekMediaView)

        }

    }

}

