package com.aotter.net.trek.ads.vast

import android.content.Context
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.View
import android.widget.ImageView
import androidx.annotation.OptIn
import androidx.media3.common.AdOverlayInfo
import androidx.media3.common.util.UnstableApi
import androidx.media3.ui.PlayerView
import androidx.multidex.MultiDex
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.trek.ads.CustomTrekMediaViewListener
import com.aotter.trek.impression.ImpressionListener
import com.aotter.trek.impression.ImpressionManager
import com.aotter.trek.impression.ImpressionProvider

class AotterPlayerView : PlayerView, ImpressionListener {

    interface AotterPlayerViewListener {

        fun onVideoReady()

    }

    private var TAG: String = AotterPlayerView::class.java.simpleName

    private var playerLoader: PlayerLoader? = null

    private var aotterPlayerViewListener: AotterPlayerViewListener? = null

    private var impressionProvider: ImpressionProvider? = null

    private var volumeImageView: ImageView = ImageView(context)

    private var clickOverlayView: View = View(context)

    private val friendlyObstructions: MutableList<AdOverlayInfo> = mutableListOf()

    constructor(context: Context) : super(context) {
        init()
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        init()
    }

    constructor(context: Context, attrs: AttributeSet?, defStyle: Int) : super(
        context, attrs, defStyle
    ) {
        init()
    }

    private fun init() {

        this.layoutParams =
            LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)

        MultiDex.install(context)

        this.useController = false

        setVolumeImageView()

    }

    fun setAotterPlayerViewListener(aotterPlayerViewListener: AotterPlayerViewListener?) {

        this.aotterPlayerViewListener = aotterPlayerViewListener

    }

    @OptIn(UnstableApi::class)
    fun initialize(
        trekNativeAd: TrekNativeAd,
        customTrekMediaViewListener: CustomTrekMediaViewListener?
    ) {

        playerLoader = PlayerLoader(
            context,
            aotterPlayerViewListener,
            this,
            volumeImageView,
            trekNativeAd,
            customTrekMediaViewListener
        ).also { playerLoader ->

            trekNativeAd.mediaSrc.getAdTagUri()?.let { uri ->

                playerLoader.initializePlayer(uri)

                launchImpressionTracker(this)

            }

        }

        setupClickOverlay(trekNativeAd)

    }

    private fun play() {

        playerLoader?.playPlayer()

    }

    fun resume() {

        impressionProvider?.resume()

    }

    fun pause() {

        playerLoader?.pausePlayer()

    }

    fun destroy() {

        playerLoader?.releasePlayer()

        destroyImpressionTracker()

    }

    private fun destroyImpressionTracker() {

        impressionProvider?.destroy()

        impressionProvider = null

    }

    private fun launchImpressionTracker(registerView: View) {

        destroyImpressionTracker()

        impressionProvider = ImpressionManager().with(registerView)

        impressionProvider?.impressionListener(this)?.apply()

    }

    override fun onVisibleRangePercent(percent: Double) {

        if (percent >= 0.5) {

            play()

        } else {

            pause()

        }

    }

    override fun getAdOverlayInfos(): MutableList<AdOverlayInfo> {

        friendlyObstructions.addAll(super.getAdOverlayInfos())

        return friendlyObstructions
    }

    private fun setVolumeImageView() {

        this.removeView(volumeImageView)

        this.addView(volumeImageView)

        val lp = LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT)

        lp.gravity = Gravity.END or Gravity.BOTTOM

        lp.setMargins(0, 0, 15, 15)

        volumeImageView.layoutParams = lp

        volumeImageView.setImageResource(android.R.drawable.ic_lock_silent_mode)

        volumeImageView.setOnClickListener {

            if (playerLoader?.player?.volume == 1.0f) {

                playerLoader?.audioFocusChangeLoader?.releaseAbandonAudioFocus()

                Log.i(TAG, "Player muted.")

            } else {

                playerLoader?.audioFocusChangeLoader?.requestAudioFocus()

                Log.i(TAG, "Player not muted.")

            }

        }

        addFriendlyObstructionView(volumeImageView)

    }

    fun addFriendlyObstructionView(friendlyObstructionView: View) {

        val adOverlayInfo =
            AdOverlayInfo.Builder(friendlyObstructionView, AdOverlayInfo.PURPOSE_NOT_VISIBLE)
                .setDetailedReason("Does not impact viewability").build()

        friendlyObstructions.add(adOverlayInfo)

    }

    private fun setupClickOverlay(trekNativeAd: TrekNativeAd) {

        // Setup following the same pattern as setVolumeImageView()
        this.removeView(clickOverlayView)

        // Add transparent overlay first (bottom layer)
        this.addView(clickOverlayView)

        // Configure overlay layout parameters
        val overlayLayoutParams = LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT)

        clickOverlayView.layoutParams = overlayLayoutParams

        // Set transparent background
        clickOverlayView.setBackgroundColor(0x00000000)

        // Set click listener for ad click trigger
        clickOverlayView.setOnClickListener {

            triggerAdClick(trekNativeAd)

        }

        // Ensure clickable
        clickOverlayView.isClickable = true

        // Re-add volumeImageView to ensure it stays on top
        this.removeView(volumeImageView)

        this.addView(volumeImageView)

        // Reconfigure volumeImageView properties since we re-added it
        val volumeLp = LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT)

        volumeLp.gravity = Gravity.END or Gravity.BOTTOM

        volumeLp.setMargins(0, 0, 15, 15)

        volumeImageView.layoutParams = volumeLp

        volumeImageView.setImageResource(android.R.drawable.ic_lock_silent_mode)

        volumeImageView.setOnClickListener {

            if (playerLoader?.player?.volume == 1.0f) {

                playerLoader?.audioFocusChangeLoader?.releaseAbandonAudioFocus()

                Log.i(TAG, "Player muted.")

            } else {

                playerLoader?.audioFocusChangeLoader?.requestAudioFocus()

                Log.i(TAG, "Player not muted.")

            }

        }
        addFriendlyObstructionView(clickOverlayView)
        addFriendlyObstructionView(volumeImageView)

    }

    private fun triggerAdClick(trekNativeAd: TrekNativeAd) {

        try {

            // Execute unified ad click logic
            trekNativeAd.executeAdClickLogic(context)

            // Notify custom video ad listener
            playerLoader?.triggerVideoAdClick()

        } catch (e: Exception) {

            Log.e(TAG, "Error triggering ad click: ${e.message}", e)

        }

    }

}