package com.aotter.net.trek.ads.vast

import android.content.ContentResolver
import android.content.Context
import android.net.Uri
import android.util.Log
import android.widget.ImageView
import androidx.annotation.OptIn
import androidx.media3.common.MediaItem
import androidx.media3.common.PlaybackException
import androidx.media3.common.Player
import androidx.media3.common.util.UnstableApi
import androidx.media3.datasource.DataSource
import androidx.media3.datasource.DefaultDataSource
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.exoplayer.ima.ImaAdsLoader
import androidx.media3.exoplayer.source.DefaultMediaSourceFactory
import androidx.media3.ui.AspectRatioFrameLayout
import androidx.media3.ui.PlayerView
import com.aotter.net.R
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.trek.ads.CustomTrekMediaViewListener
import com.google.ads.interactivemedia.v3.api.AdEvent
import com.google.ads.interactivemedia.v3.api.ImaSdkFactory
import java.util.Locale

class PlayerLoader(
    private val context: Context,
    private val aotterPlayerViewListener: AotterPlayerView.AotterPlayerViewListener?,
    private var styledPlayerView: PlayerView,
    private val volumeImageView: ImageView,
    private val trekNativeAd: TrekNativeAd,
    private val customTrekMediaViewListener: CustomTrekMediaViewListener?
) :
    Player.Listener {

    private var TAG: String = PlayerLoader::class.java.simpleName

    var player: ExoPlayer? = null
        get() = field

    var imaAdsLoader: ImaAdsLoader? = null
        get() = field

    var audioFocusChangeLoader: AudioFocusChangeLoader? = null
        get() = field

    private var adTagUri: Uri? = null

    override fun onPlayerError(error: PlaybackException) {
        super.onPlayerError(error)

        audioFocusChangeLoader?.releaseAbandonAudioFocus()

    }

    @OptIn(UnstableApi::class)
    override fun onPlaybackStateChanged(state: Int) {
        super.onPlaybackStateChanged(state)

        when (state) {
            ExoPlayer.STATE_READY -> aotterPlayerViewListener?.onVideoReady()

            // Release audio focus on loading error
            ExoPlayer.STATE_IDLE -> {

                releasePlayer()

                adTagUri?.let { uri ->
                    initializePlayer(uri)
                }

            }

            // Release audio focus when playback ends
            ExoPlayer.STATE_ENDED -> {
                customTrekMediaViewListener?.onVideoAdEnded()
                releasePlayer()

                adTagUri?.let { uri ->
                    initializePlayer(uri)
                }

            }

        }

    }

    @OptIn(UnstableApi::class)
    fun initializePlayer(adTagUri: Uri) {

        this.adTagUri = adTagUri

        val imaSdkSettings = ImaSdkFactory.getInstance().createImaSdkSettings()

        imaSdkSettings.language = getLanguage()

        imaAdsLoader =
            ImaAdsLoader.Builder(context.applicationContext).setImaSdkSettings(imaSdkSettings)
                .setAdEventListener { event ->
                    when (event.type) {
                        AdEvent.AdEventType.CLICKED -> {
                            customTrekMediaViewListener?.onVideoAdClick(trekNativeAd.originalUrl)
                        }

                        else -> {}
                    }
                }
                .build()

        styledPlayerView.resizeMode = AspectRatioFrameLayout.RESIZE_MODE_FILL

        // Set up the factory for media sources, passing the ads loader and ad view providers.
        val dataSourceFactory: DataSource.Factory =
            DefaultDataSource.Factory(context.applicationContext)

        val mediaSourceFactory = DefaultMediaSourceFactory(dataSourceFactory)
            .setLocalAdInsertionComponents(
                { unusedAdTagUri: MediaItem.AdsConfiguration? ->
                    imaAdsLoader
                },
                styledPlayerView
            )

        // Create an ExoPlayer and set it as the player for content and ads.
        player =
            ExoPlayer.Builder(context.applicationContext).setMediaSourceFactory(mediaSourceFactory)
                .build()

        styledPlayerView.player = player

        imaAdsLoader?.setPlayer(player)

        val uri = Uri.Builder()
            .scheme(ContentResolver.SCHEME_ANDROID_RESOURCE)
            .authority(context.packageName)
            .appendPath("${R.raw.black_video}")
            .build()

        val mediaItem = MediaItem.Builder()
            .setUri(uri)
            .setAdsConfiguration(MediaItem.AdsConfiguration.Builder(adTagUri).build())
            .build()

        player?.apply {

            // Prepare the content and ad to be played with the ExoPlayer.
            setMediaItem(mediaItem)

            addListener(this@PlayerLoader)

            prepare()

            // Set PlayWhenReady. If true, content and ads will autoplay.
            playWhenReady = false

            audioFocusChangeLoader =
                AudioFocusChangeLoader(context, player, volumeImageView)

            Log.i(TAG, "Player initialized.")

        }

    }

    fun playPlayer() {

        if (player?.playWhenReady == false) {

            player?.playWhenReady = true

            Log.i(TAG, "Player played.")

        }

    }

    fun pausePlayer() {

        if (player?.playWhenReady == true) {

            audioFocusChangeLoader?.releaseAbandonAudioFocus()

            player?.playWhenReady = false

            Log.i(TAG, "Player paused.")

        }

    }

    fun releasePlayer() {

        pausePlayer()

        audioFocusChangeLoader?.destroy()

        imaAdsLoader?.let {

            it.setPlayer(null)

            it.release()

            imaAdsLoader = null

        }

        player?.removeListener(this)

        player?.release()

        player = null

        Log.i(TAG, "Player released.")

    }

    private fun getLanguage(): String {

        val language = Locale.getDefault().language

        val country = Locale.getDefault().country

        return if (language == "zh" && country == "TW" || language == "zh" && country == "tw") {

            "${language}_tw"

        } else {

            language

        }

    }

    fun triggerVideoAdClick() {

        try {

            customTrekMediaViewListener?.onVideoAdClick(trekNativeAd.originalUrl)

        } catch (e: Exception) {

            Log.e(TAG, "Error triggering video ad click: ${e.message}", e)

        }

    }

}