package com.aotter.net.trek.ads.webview

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.constraintlayout.widget.ConstraintSet
import com.aotter.net.databinding.ViewAotterWebBinding
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.trek.ads.base.BaseWebView
import com.aotter.net.trek.ads.base.TrekBaseView
import com.aotter.net.trek.sealed.SrcShowField
import com.aotter.trek.impression.ImpressionListener
import com.aotter.trek.impression.ImpressionManager


class AotterWebView : TrekBaseView, ImpressionListener {

    interface AotterWebViewListener {

        fun onSendUrlClc(urlClcString: String)

        fun onPageLoaded()

        fun onClicked()

    }

    private val TAG: String = AotterWebView::class.java.simpleName

    private var aotterWebViewListener: AotterWebViewListener? = null

    private var webview: BaseWebView? = null

    constructor(context: Context) : super(context) {
        init()
    }

    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
        init()
    }

    constructor(context: Context, attrs: AttributeSet?, defStyle: Int) : super(
        context, attrs, defStyle
    ) {
        init()
    }

    lateinit var viewBinding: ViewAotterWebBinding

    private var webViewLoader: WebViewLoader? = null

    private fun init() {

        viewBinding = ViewAotterWebBinding.inflate(LayoutInflater.from(context), this, true)

        viewBinding.root.layoutParams =
            LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT).apply {
                bottomToBottom = ConstraintSet.PARENT_ID
                endToEnd = ConstraintSet.PARENT_ID
                startToStart = ConstraintSet.PARENT_ID
                topToTop = ConstraintSet.PARENT_ID
            }

    }

    fun setAotterWebViewListener(aotterWebViewListener: AotterWebViewListener) {

        this.aotterWebViewListener = aotterWebViewListener

    }

    fun resume() {

        impressionProvider?.resume()

    }

    fun pause() {

        webViewLoader?.pauseWebView()

    }

    fun destroy() {

        destroyImpressionTracker()

        webViewLoader?.destroyWebView()

        webview?.loadDataWithBaseURL(null, "", "text/html", "utf-8", null)

        webview?.clearHistory()

        (webview?.parent as? ViewGroup)?.removeView(webview)

        webview?.destroy()

        webview = null

        removeAllViews()

    }

    private fun initWebView(trekNativeAd: TrekNativeAd?) {

        trekNativeAd?.impressionListener = this

        destroy()

        webview = BaseWebView(context).also { webView ->

            val lp = LayoutParams(
                LayoutParams.MATCH_PARENT,
                LayoutParams.WRAP_CONTENT
            ).apply {
                bottomToBottom = ConstraintSet.PARENT_ID
                endToEnd = ConstraintSet.PARENT_ID
                startToStart = ConstraintSet.PARENT_ID
                topToTop = ConstraintSet.PARENT_ID
            }

            this@AotterWebView.addView(webView, lp)

            webViewLoader = WebViewLoader(context, aotterWebViewListener, webView, trekNativeAd)

        }

    }

    fun loadHtmlAd(loadHtmlString: String) {

        if (loadHtmlString.isEmpty()) return

        initWebView(null)

        webview?.loadDataWithBaseURL(
            null, loadHtmlString, "text/html", "UTF-8", null
        )

    }


    fun loadUrlAd(loadUrlString: String) {

        if (loadUrlString.isEmpty()) return

        initWebView(null)

        webview?.loadUrl(loadUrlString)

    }

    fun loadAd(trekNativeAd: TrekNativeAd) {

        initWebView(trekNativeAd)

        val adm = trekNativeAd.mediaSrc.adm

        val widgetUrl = trekNativeAd.mediaSrc.widgetUrl

        val widgetHtml = trekNativeAd.mediaSrc.widgetHtml

        when (trekNativeAd.mediaSrc.srcShowField) {
            SrcShowField.HTML.type -> webview?.loadDataWithBaseURL(
                null, widgetHtml ?: "", "text/html", "UTF-8", null
            )
            SrcShowField.ADM.type -> webview?.loadDataWithBaseURL(
                null, adm ?: "", "text/html", "UTF-8", null
            )
            SrcShowField.URL.type -> webview?.loadUrl(widgetUrl ?: "")
            else -> webview?.loadUrl(widgetUrl ?: "")
        }

        launchImpressionTracker(viewBinding.root)

    }

    fun notifyCatrunImpressionOccurred() {
        webViewLoader?.notifyCatrunImpressionOccurred()
    }

    private fun destroyImpressionTracker() {

        impressionProvider?.destroy()

        impressionProvider = null

    }

    private fun launchImpressionTracker(registerView: View) {

        destroyImpressionTracker()

        impressionProvider = ImpressionManager().with(registerView)

        impressionProvider?.impressionListener(this)?.apply()

    }

    override fun onRatio(ratio: Double) {

        webViewLoader?.calculateRollingParallax((ratio * 100).toInt())

    }

    override fun onVisibleRangePercent(percent: Double) {

        if (percent > 0.0 && percent <= 1.0) {

            webViewLoader?.resumeWebView()

        } else {

            pause()

        }

        webViewLoader?.intersecting(percent)

    }

}