package com.aotter.net.trek.ads.webview

import android.content.Context
import android.content.Intent
import android.util.Log
import android.webkit.JavascriptInterface
import android.webkit.WebView
import androidx.fragment.app.FragmentActivity
import com.aotter.net.dto.trek.response.CatRunJavascriptInterfaceDto
import com.aotter.net.utils.TrekSdkSettingsUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import kotlinx.serialization.decodeFromString

class CatRunInitBridge(
    private val context: Context,
    private val aotterWebViewListener: AotterWebView.AotterWebViewListener?,
    private val webview: WebView,
) {

    private val TAG: String = CatRunInitBridge::class.java.simpleName

    companion object {

        private const val POPUP_WEB_VIEW_DIALOG_FRAGMENT_TAG = "PopupWebViewDialogFragment"

        private const val CATRUN_INIT_ACTION_BRIDGE = "CatRunInitActionBridge"


    }

    private var scope = CoroutineScope(Dispatchers.Main.immediate + SupervisorJob())

    private var popupWebViewDialogFragment: PopupWebViewDialogFragment? = null

    var isParallax: Boolean = true
        get() = field

    private var sensorsLoader: SensorsLoader? = null

    init {

        webview.removeJavascriptInterface(CATRUN_INIT_ACTION_BRIDGE)

        webview.addJavascriptInterface(CatRunActionBridge(), CATRUN_INIT_ACTION_BRIDGE)

    }

    @JavascriptInterface
    fun initBridge() {

        scope.launch {

            Log.i(TAG, "initBridge")

            sensorsLoader = SensorsLoader(context, webview)

            webview.evaluateJavascript("javascript:window.WebViewBridge.on('send', function(message) { ${CATRUN_INIT_ACTION_BRIDGE}.initActionBridge(message);})") {}

        }

    }

    private fun showWebViewPage(urlString: String?) {

        if (urlString == null || urlString == "") return

        (this.context as? FragmentActivity)?.let { fragmentActivity ->

            popupWebViewDialogFragment = PopupWebViewDialogFragment.newInstance(urlString).apply {

                show(
                    fragmentActivity.supportFragmentManager,
                    POPUP_WEB_VIEW_DIALOG_FRAGMENT_TAG
                )

            }

        } ?: kotlin.run {

            val intent = Intent()

            intent.putExtra(PopupWebViewActivity.URL, urlString)

            intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK

            intent.setClass(this.context, PopupWebViewActivity::class.java)

            this.context.startActivity(intent)

        }

    }

    private fun sendUrlClc(urlClcString: String?) {

        if (urlClcString == null || urlClcString == "") return

        aotterWebViewListener?.onSendUrlClc(urlClcString)

    }

    fun resume() {

        sensorsLoader?.register()

    }

    fun pause() {

        sensorsLoader?.unregister()

    }

    fun destroy() {

        sensorsLoader?.destroy()

        popupWebViewDialogFragment?.dismissAllowingStateLoss()

        popupWebViewDialogFragment = null

    }

    private inner class CatRunActionBridge() {

        @JavascriptInterface
        fun initActionBridge(message: String) {

            Log.i(TAG, "action : $message")

            try {

                scope.launch {

                    val catRunJavascriptInterfaceDto =
                        TrekSdkSettingsUtils.json.decodeFromString<CatRunJavascriptInterfaceDto>(
                            message
                        )

                    when (catRunJavascriptInterfaceDto.message.action) {

                        CatRunActionType.LOADED.value -> {

                            isParallax =
                                catRunJavascriptInterfaceDto.message.payload?.events?.PARALLAX?.enabled
                                    ?: true

                            val accelerometer =
                                catRunJavascriptInterfaceDto.message.payload?.events?.ACCELEROMETER?.enabled
                                    ?: false

                            val gyroscope =
                                catRunJavascriptInterfaceDto.message.payload?.events?.GYROSCOPE?.enabled
                                    ?: false

                            if (accelerometer) {

                                sensorsLoader?.initAccelerometer()

                            }

                            if (gyroscope) {

                                sensorsLoader?.initGyroscope()

                            }

                            aotterWebViewListener?.onPageLoaded()

                        }

                        CatRunActionType.POPUP.value -> {

                            showWebViewPage(catRunJavascriptInterfaceDto.message.payload?.url)

                        }

                        CatRunActionType.CLOSE.value -> {

                            popupWebViewDialogFragment?.dismissAllowingStateLoss()

                        }

                        CatRunActionType.CLICK.value -> {

                            showWebViewPage(catRunJavascriptInterfaceDto.message.payload?.url)

                        }

                        CatRunActionType.OPEN_EXTERNAL_BROWSER.value -> {

                            TrekSdkSettingsUtils.openBrowser(
                                context, catRunJavascriptInterfaceDto.message.payload?.url ?: ""
                            )

                        }
                        CatRunActionType.UNKNOWN.value -> {
                        }
                    }

                    //當 action 不等於 null & 空字串 且 url_clc 不等於 null & 空字串
                    if (catRunJavascriptInterfaceDto.message.action.isNotEmpty() && catRunJavascriptInterfaceDto.message.url_clc?.isNotEmpty() == true) {

                        aotterWebViewListener?.onClicked()

                    }

                    sendUrlClc(catRunJavascriptInterfaceDto.message.url_clc)

                }

            } catch (e: Exception) {
                Log.e(TAG, e.toString())
            }

        }

    }

}