package com.aotter.net.trek.ads.webview

import android.content.Context
import android.graphics.Bitmap
import android.util.Log
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.trek.sealed.SrcShowField
import com.aotter.net.utils.TrekSdkSettingsUtils
import java.util.concurrent.atomic.AtomicBoolean

class WebViewLoader(
    private val context: Context,
    private val aotterWebViewListener: AotterWebView.AotterWebViewListener?,
    private val webview: WebView,
    private val trekNativeAd: TrekNativeAd?
) : WebViewClient() {

    private val TAG: String = WebViewLoader::class.java.simpleName

    companion object {

        private const val CATRUN_INIT_BRIDGE = "CatRunInitBridge"

    }

    private var isWebViewFinished = AtomicBoolean(false)

    private var catRunInitBridge: CatRunInitBridge? = null

    init {

        webview.let { webview ->

            webview.webViewClient = this

            catRunInitBridge = CatRunInitBridge(
                context, aotterWebViewListener, webview
            ).also { catRunInitBridge ->

                webview.removeJavascriptInterface(CATRUN_INIT_BRIDGE)

                webview.addJavascriptInterface(catRunInitBridge, CATRUN_INIT_BRIDGE)

            }

        }

    }

    override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {

        isWebViewFinished.set(false)

        webview.evaluateJavascript("javascript:window.bridgeInit = () => { ${CATRUN_INIT_BRIDGE}.initBridge(); }") { result -> }

    }

    override fun onPageFinished(view: WebView?, url: String?) {
        super.onPageFinished(view, url)

        if (isWebViewFinished.getAndSet(true)) return

        val showField = trekNativeAd?.mediaSrc?.srcShowField

        if (showField == SrcShowField.ADM.type) {
            aotterWebViewListener?.onPageLoaded()
        }

    }

    override fun shouldOverrideUrlLoading(
        view: WebView?, request: WebResourceRequest?
    ): Boolean {

        request?.url?.apply {

            val urlString = this.toString()

            if (urlString.startsWith("http")) {

                TrekSdkSettingsUtils.openBrowser(
                    context, urlString
                )

                aotterWebViewListener?.onClicked()

            }

        }

        return true

    }

    fun resumeWebView() {

        catRunInitBridge?.resume()

    }

    fun pauseWebView() {

        catRunInitBridge?.pause()

    }

    fun destroyWebView() {

        catRunInitBridge?.destroy()

    }

    //通知catrun 要記錄 imp
    fun notifyCatrunImpressionOccurred() {

        Log.i(TAG, "WebView impression.")

        webview.evaluateJavascript("javascript:window.WebViewBridge.emit('ON_DISPLAY')") {}

    }

    fun intersecting(percent: Double) {

        val isIntersecting = percent > 0.0

        webview.evaluateJavascript("javascript:window.WebViewBridge.emit('INTERSECTION', { payload: { isIntersecting: $isIntersecting, intersectionRatio: $percent } })") {}

    }

    //計算滾動視差
    fun calculateRollingParallax(position: Int) {

        if (catRunInitBridge?.isParallax == false) return

        webview.evaluateJavascript("javascript:window.WebViewBridge.emit('PARALLAX', { payload: {position: $position} })") {}

    }

}