package com.aotter.net.utils

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.content.pm.PackageManager
import android.os.*
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import java.io.IOException
import java.util.concurrent.LinkedBlockingQueue


object AdvertisingIdClientUtils {

    fun getAdvertisingIdInfoCase1(context: Context): String {

        return try {

            val advertisingIdInfo =
                AdvertisingIdClient.getAdvertisingIdInfo(context.applicationContext)

            // The limit ad tracking setting. It is true if the user has limit ad tracking enabled. False, otherwise.
            if (!advertisingIdInfo.isLimitAdTrackingEnabled) {

                advertisingIdInfo.id ?: getAdvertisingIdInfoCase2(context.applicationContext)

            } else {

                getAdvertisingIdInfoCase2(context.applicationContext)

            }

        } catch (e: Exception) {

            getAdvertisingIdInfoCase2(context.applicationContext)

        }

    }

    @Throws(Exception::class)
    fun getAdvertisingIdInfoCase2(context: Context): String {
        check(Looper.myLooper() != Looper.getMainLooper()) { "Cannot be called from the main thread" }
        try {
            val pm: PackageManager = context.packageManager
            pm.getPackageInfo("com.android.vending", 0)
        } catch (e: Exception) {
            throw e
        }
        val connection = AdvertisingConnection()
        val intent = Intent(
            "com.google.android.gms.ads.identifier.service.START"
        )
        intent.setPackage("com.google.android.gms")
        if (context.bindService(intent, connection, Context.BIND_AUTO_CREATE)) {
            return try {
                val adInterface = AdvertisingInterface(
                    connection.binder
                )
                val adInfo = AdInfo(
                    adInterface.id,
                    adInterface.isLimitAdTrackingEnabled(true)
                )
                adInfo.id ?: ""
            } catch (exception: Exception) {
                throw exception
            } finally {
                context.unbindService(connection)
            }
        }
        throw IOException("Google Play connection failed")
    }

    class AdInfo internal constructor(val id: String?, val isLimitAdTrackingEnabled: Boolean)
    private class AdvertisingConnection : ServiceConnection {
        var retrieved = false
        private val queue: LinkedBlockingQueue<IBinder> = LinkedBlockingQueue<IBinder>(
            1
        )

        override fun onServiceConnected(name: ComponentName, service: IBinder) {
            try {
                queue.put(service)
            } catch (localInterruptedException: InterruptedException) {
            }
        }

        override fun onServiceDisconnected(name: ComponentName) {}

        @get:Throws(InterruptedException::class)
        val binder: IBinder
            get() {
                check(!retrieved)
                retrieved = true
                return queue.take()
            }
    }

    private class AdvertisingInterface(private val binder: IBinder) : IInterface {
        override fun asBinder(): IBinder {
            return binder
        }

        @get:Throws(RemoteException::class)
        val id: String?
            get() {
                val data = Parcel.obtain()
                val reply = Parcel.obtain()
                return try {
                    data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService")
                    binder.transact(1, data, reply, 0)
                    reply.readException()
                    reply.readString()
                } finally {
                    reply.recycle()
                    data.recycle()
                }
            }

        @Throws(RemoteException::class)
        fun isLimitAdTrackingEnabled(paramBoolean: Boolean): Boolean {
            val data = Parcel.obtain()
            val reply = Parcel.obtain()
            return try {
                data.writeInterfaceToken("com.google.android.gms.ads.identifier.internal.IAdvertisingIdService")
                data.writeInt(if (paramBoolean) 1 else 0)
                binder.transact(2, data, reply, 0)
                reply.readException()
                0 != reply.readInt()
            } finally {
                reply.recycle()
                data.recycle()
            }
        }
    }
}