package com.aotter.net.utils

import android.content.Context
import android.net.ConnectivityManager
import android.os.Build
import android.provider.Settings
import android.telephony.TelephonyManager
import android.util.Log
import com.aotter.net.dto.Device
import com.aotter.net.extension.getVersionName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

object DeviceUtils {

    private val TAG: String = DeviceUtils::class.java.simpleName

    private val networkTypeMap = mapOf(
        TelephonyManager.NETWORK_TYPE_GPRS to "2G",
        TelephonyManager.NETWORK_TYPE_EDGE to "2G",
        TelephonyManager.NETWORK_TYPE_CDMA to "2G",
        TelephonyManager.NETWORK_TYPE_IDEN to "2G",
        TelephonyManager.NETWORK_TYPE_UMTS to "3G",
        TelephonyManager.NETWORK_TYPE_EVDO_0 to "3G",
        TelephonyManager.NETWORK_TYPE_EVDO_A to "3G",
        TelephonyManager.NETWORK_TYPE_HSDPA to "3G",
        TelephonyManager.NETWORK_TYPE_HSUPA to "3G",
        TelephonyManager.NETWORK_TYPE_HSPA to "3G",
        TelephonyManager.NETWORK_TYPE_EVDO_B to "3G",
        TelephonyManager.NETWORK_TYPE_EHRPD to "3G",
        TelephonyManager.NETWORK_TYPE_TD_SCDMA to "3G",
        TelephonyManager.NETWORK_TYPE_LTE to "4G",
        TelephonyManager.NETWORK_TYPE_NR to "5G"
    )

    private var device: Device = Device()

    fun getDevice() = device

    fun setDevice(context: Context) {

        val adId = getAdId(context)
        val deviceId = getDeviceId(context)
        val deviceAppVersionName = context.getVersionName()
        val deviceModel = Build.MODEL
        val deviceOsVersion = Build.VERSION.RELEASE
        val deviceBrand = Build.MANUFACTURER
        val deviceNetWork = getDeviceNetworkCondition(context)
        val deviceOperator = getDeviceOperator(context)
        val sdkInt = Build.VERSION.SDK_INT

        val deviceBo = Device(
            adId,
            deviceAppVersionName,
            deviceBrand,
            deviceId,
            deviceModel,
            deviceNetWork,
            deviceOperator,
            deviceOsVersion,
            sdkInt
        )

        device = deviceBo

    }

    fun getDeviceNetworkCondition(context: Context): String {

        val unknown = "UNKNOWN"

        try {

            val connectivityManager =
                (context.getSystemService(Context.CONNECTIVITY_SERVICE) as? ConnectivityManager)
                    ?: return unknown

            connectivityManager.allNetworkInfo.firstOrNull { it.isConnected }?.let {
                return when (it.type) {
                    ConnectivityManager.TYPE_WIFI -> "WIFI"
                    ConnectivityManager.TYPE_MOBILE -> networkTypeMap[it.subtype] ?: unknown
                    else -> unknown
                }
            } ?: return unknown

        } catch (e: Exception) {
            return unknown
        }

    }

    fun getDeviceOperator(context: Context): String {

        return try {

            val telephonyManager =
                context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager

            telephonyManager.networkOperator

        } catch (e: Exception) {
            ""
        }

    }

    fun getDeviceId(context: Context): String {

        return try {

            Settings.Secure.getString(
                context.contentResolver,
                Settings.Secure.ANDROID_ID
            )

        } catch (e: Exception) {
            ""
        }

    }

    fun getAdId(context: Context): String {

        CoroutineScope(Dispatchers.IO).launch {

            try {

                PreferencesDataStoreUtils.setAdId(
                    context,
                    AdvertisingIdClientUtils.getAdvertisingIdInfoCase1(context.applicationContext)
                )

                DeviceUtils.getDevice().adID = PreferencesDataStoreUtils.getAdId(context)

                Log.i("getAdId", "adId : ${DeviceUtils.getDevice().adID}")


            } catch (e: Exception) {
                Log.e("getAdId", e.toString())
            }

        }

        val adId = PreferencesDataStoreUtils.getAdId(context)

        return if (adId != "") {
            adId
        } else {
            ""
        }

    }


}

