package com.aotter.net.utils

import android.content.Context
import com.aotter.net.controller.eids.EidsController
import com.aotter.net.dto.eids.request.EidsGenerationRequestBody
import com.aotter.net.dto.eids.request.EidsRefreshRequestBody
import com.aotter.net.dto.eids.response.UserEids
import com.aotter.net.model.repository.eids.EidsRepository
import com.aotter.net.network.Resource
import com.aotter.net.network.RetrofitBuilder
import com.aotter.net.trek.TrekAds
import com.aotter.net.trek.sealed.EidsRemoveByType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean


/**
 * About Eids
 *
 * 2024/05/07 Create By Charlotte
 *
 * Reference: https://github.com/aotter/aotter-trek-android/wiki/Eids
 *
 *
 * Initialization
 *
 * 1. During app launch, check the refresh_expires of the existing EIDs. If it has expired, remove it.
 * 2. Call the generate API, including unexpired EIDs in the exists_eid_type array.
 * 3. Add new tokens from the generate response to local data.
 *
 *
 * refresh
 *
 * 1. Before calling the refresh API, check the refresh_expires of the EIDs. If it has expired, remove it.
 * 2. Check the refresh_from of the EIDs; if it has expired, initiate a refresh.
 *
 *
 * When setting UserInfo, the email or phone differs from the local data.
 *
 * 1. Check the EIDs and remove entries associated with hashed_email or hashed_phone.
 * 2. After removing the old token(EID), either generate new tokens(EID) or take no action.
 *
 *
 *  When fetching ads, UserInfo's eidsItems should adhere to the following conditions,
 *  which are checked before fetching an ad:
 *
 *  1. If there are no Eids, UserInfo's eidsItems should remain empty.
 *  2. If identity_expires, refresh_from, and identity_expires are not expired, UserInfo's eidsItems should contain the EID.
 *  3. If identity_expires is not expired, UserInfo's eidsItems should contain the eid; refresh_from should be checked for a refresh.
 *  4. If EID's identity_expires has expired, UserInfo's eidsItems should not contain this data, and a refresh should be awaited.
 *  5. If EID's refresh_expires has expired, remove it. This means UserInfo's eidsItems should not contain this data and cannot be refreshed.
 *
 */

object EidsUtils {

    private val TAG: String = EidsUtils::class.java.simpleName

    private const val AD_TYPE: String = "gaid"

    private var eidsController: EidsController? = null

    private val eidsHashMap: ConcurrentHashMap<String, UserEids> =
        ConcurrentHashMap() //runtime newest data

    private var lastRefreshTimeStamp: Long = 0L

    private val isInitialized = AtomicBoolean(false)

    private val ioScope = CoroutineScope(Dispatchers.IO)

    /**
     * Initialization.
     * This function is called by TrekAds when the app launches.
     * Executes asynchronously to avoid blocking the main thread.
     */
    fun init(context: Context) {
        if (isInitialized.getAndSet(true)) {
            return // Already initialized
        }

        initEidsController()

        // Execute initialization asynchronously to avoid ANR
        ioScope.launch {
            try {
                initData(context)
                generateEids(context)
            } catch (e: Exception) {
                // Log error but don't crash
                android.util.Log.e(TAG, "Error initializing EIDS: ${e.message}", e)
            }
        }
    }

    /**
     * Only for TrekAdLoader.
     * Retrieves a valid EID list before each ad fetch.
     */
    fun getEidList(): List<UserEids> {
        val now = System.currentTimeMillis()
        TrekAds.getApplicationContext()?.let { context ->
            checkEidsIfNeedRefresh(context)
        }
        return eidsHashMap.values.filter { !it.identityTimeout(now) }
    }

    /**
     * Only for UserInfoUtils
     * Checks if the email or phone has changed. If there's a change, removes the old token and generates a new token.
     */
    fun updateEidsIfUserInfoChanged(types: List<EidsRemoveByType>) {
        eidsHashMap
            .filter { it.value.needRemove(types) }
            .forEach { eidsHashMap.remove(it.key) }

        TrekAds.getApplicationContext()?.let { context ->
            generateEids(context)
        }
    }

    fun clearUserInfo() {
        eidsHashMap.clear()
    }

    private fun initEidsController() {

        eidsController ?: run {
            eidsController =
                EidsController(EidsRepository(RetrofitBuilder.createEidsService()))
        }

    }

    private suspend fun initData(context: Context) {
        eidsHashMap.clear()
        val localUserEids = getLocalUserEids(context) ?: emptyList()
        val now = System.currentTimeMillis()
        localUserEids
            .filter { !it.refreshTimeout(now) }
            .forEach { eidsHashMap[it.eidType] = it }
    }

    private fun generateEids(context: Context) {
        val request = getEidsGenerationRequestBody(context)
        eidsController?.generateEids(request, onGenerateListener)
    }

    private var onGenerateListener = object : EidsController.OnGenerateListener {

        override fun onGenerate(resource: Resource<ArrayList<UserEids>>) {
            resource.data?.forEach {
                eidsHashMap[it.eidType] = it
            }
            saveUserEids()
        }

    }

    private fun refreshEids(context: Context, userEids: UserEids) {
        val request = getEidsRefreshRequestBody(context, userEids)
        eidsController?.refreshEids(request, onRefreshListener)
    }

    private var onRefreshListener = object : EidsController.OnRefreshListener {

        override fun onRefresh(resource: Resource<UserEids>) {
            resource.data?.let { response ->
                eidsHashMap[response.eidType] = response
                saveUserEids()
            }
        }

    }

    private fun saveUserEids() {
        val json = Json.encodeToString(eidsHashMap.values.toList())
        TrekAds.getApplicationContext()?.let { context ->
            // Save asynchronously to avoid blocking
            ioScope.launch {
                try {
                    PreferencesDataStoreUtils.setUserEids(context, json)
                } catch (e: Exception) {
                    android.util.Log.e(TAG, "Error saving EIDS: ${e.message}", e)
                }
            }
        }
    }

    private suspend fun getLocalUserEids(context: Context): List<UserEids>? {
        return try {
            PreferencesDataStoreUtils.getHashEids(context)
                ?.let { TrekSdkSettingsUtils.json.decodeFromString<List<UserEids>>(it) }
        } catch (e: Exception) {
            null
        }
    }

    /**
     * Generate the RequestBody for generating EIDs.
     * Obtain a list of valid existing Eid types through getExistEidTypeList.
     */
    private fun getEidsGenerationRequestBody(context: Context): EidsGenerationRequestBody {

        val existEidTypeList = eidsHashMap.mapNotNull { it.value.eidType }
        val bundleId = AppInfoUtils.getAppId()
        val ver = AppInfoUtils.getAppVersion()
        val ua = TrekSdkSettingsUtils.getDefaultUserAgent()
        val adId = DeviceUtils.getAdId(context)
        val hashEmail = UserInfoUtils.getUserInfo().email.ifEmpty { null }
        val hashPhone = UserInfoUtils.getUserInfo().phone.ifEmpty { null }
        val deviceID = DeviceUtils.getDeviceId(context)

        return EidsGenerationRequestBody(
            existEidTypeList,
            bundleId,
            ver,
            ua,
            adId,
            AD_TYPE,
            hashEmail,
            hashPhone,
            deviceID
        )

    }

    /**
     * Check if Eids data needs to be refreshed.
     * If the current time is greater than refresh_from, perform a refresh.
     */
    private fun checkEidsIfNeedRefresh(context: Context) {

        val now = System.currentTimeMillis()

        updateEidsIfRefreshExpired()

        if ((now - lastRefreshTimeStamp) < 1000L) return // Prevent unnecessary calls to the refresh API caused by multiple fetch ad requests

        lastRefreshTimeStamp = now

        eidsHashMap.values
            .filter { it.needRefresh(now) }
            .forEach { refreshEids(context, it) }
        
    }

    private fun getEidsRefreshRequestBody(
        context: Context,
        userEids: UserEids
    ): EidsRefreshRequestBody {

        val bundleId = AppInfoUtils.getAppId()
        val ver = AppInfoUtils.getAppVersion()
        val ua = TrekSdkSettingsUtils.getDefaultUserAgent()
        val adId = DeviceUtils.getAdId(context)
        val hashEmail = userEids.hashedEmail
        val hashPhone = userEids.hashedPhone
        val currentHashEmail = UserInfoUtils.getUserInfo().email.ifEmpty { null }
        val currentHashPhone = UserInfoUtils.getUserInfo().phone.ifEmpty { null }
        val eidType = userEids.eidType
        val refreshToken = userEids.refreshToken
        val refreshKey = userEids.refreshKey

        return EidsRefreshRequestBody(
            bundleId,
            ver,
            ua,
            adId,
            AD_TYPE,
            hashEmail,
            hashPhone,
            currentHashEmail,
            currentHashPhone,
            eidType,
            refreshToken,
            refreshKey
        )

    }

    private fun updateEidsIfRefreshExpired() {
        val now = System.currentTimeMillis()

        eidsHashMap.filter { it.value.refreshTimeout(now) }
            .forEach { eidsHashMap.remove(it.key) }
    }

}