package com.aotter.net.utils

import android.content.Context
import com.aotter.net.dto.EidsItem
import com.aotter.net.dto.Uid
import com.aotter.net.dto.User
import com.aotter.net.extension.hashEmailWithSHA256
import com.aotter.net.extension.hashPhoneWithSHA256
import com.aotter.net.extension.isValidEmail
import com.aotter.net.extension.isValidPhone
import com.aotter.net.trek.TrekAds
import com.aotter.net.trek.sealed.EidsRemoveByType
import kotlinx.coroutines.runBlocking

object UserInfoUtils {

    private val TAG: String = UserInfoUtils::class.java.simpleName

    private var userInfo: User = User()
    private var trackerUserInfo: User = User()

    fun getUserInfo(): User = userInfo
    fun getTrackerUserInfo(): User = trackerUserInfo

    fun init(context: Context) {
        updateIfUserInfoChanged(context)
    }

    /**
     * UserInfoUtils.setUserInfo might be called before UserInfoUtils.init.
     * If setUserInfo is called before init, there is no context to check local data.
     * If the email or phone is invalid, do not store the value in memory. Instead,
     * retrieve local data when updateIfUserInfoChanged is called with context.
     */
    fun setUserInfo(userInfo: User) {

        userInfo.apply {
            email = if (email.isValidEmail()) {
                email.hashEmailWithSHA256()
            } else {
                ""
            }
            phone = if (phone.isValidPhone()) {
                phone.hashPhoneWithSHA256()
            } else {
                ""
            }
        }

        this.userInfo = userInfo

        TrekAds.getApplicationContext()?.let {
            updateIfUserInfoChanged(it)
        }
    }

    fun setTrackerUserInfo(userInfo: User) {

        userInfo.email = userInfo.email.hashEmailWithSHA256()
        userInfo.phone = userInfo.phone.hashPhoneWithSHA256()

        trackerUserInfo = userInfo

    }

    fun setUserEidsItemList() {
        val eidsItemList = EidsUtils.getEidList()
            .groupBy { it.source }
            .map {
                EidsItem(
                    it.value.map { item -> Uid(item.advertisingToken, item.ext) }.toMutableList(),
                    it.key,
                )
            }
        userInfo.setEidsItems(eidsItemList)
    }

    fun clearUserInfo(context: Context) {

        userInfo = User()

        EidsUtils.clearUserInfo()

        runBlocking {

            PreferencesDataStoreUtils.clearSavedHashData(context)

        }

    }

    private fun setHashEmail(context: Context, email: String) {

        runBlocking {

            PreferencesDataStoreUtils.setHashEmail(context, email)

        }
    }

    private fun setHashPhone(context: Context, phone: String) {

        runBlocking {

            PreferencesDataStoreUtils.setHashPhone(context, phone)

        }
    }

    /**
     * If userInfo is empty, retrieve data from local storage.
     * If userInfo is not empty, compare it with the local data.
     * If userInfo differs from the local data, save the new values and notify EidsUtils that userInfo has changed.
     */
    private fun updateIfUserInfoChanged(context: Context) {

        val savedEmail = PreferencesDataStoreUtils.getHashEmail(context)
        val savedPhone = PreferencesDataStoreUtils.getHashPhone(context)

        if (userInfo.email.isEmpty()) userInfo.email = savedEmail
        if (userInfo.phone.isEmpty()) userInfo.phone = savedPhone

        val eidsRemoveByUserInfoChangedList = mutableListOf<EidsRemoveByType>()

        if (userInfo.email != savedEmail) {
            setHashEmail(context, userInfo.email)
            eidsRemoveByUserInfoChangedList.add(EidsRemoveByType.HASH_EMAIL)
        }

        if (userInfo.phone != savedPhone) {
            setHashPhone(context, userInfo.phone)
            eidsRemoveByUserInfoChangedList.add(EidsRemoveByType.HASH_PHONE)
        }

        if (eidsRemoveByUserInfoChangedList.isNotEmpty()) {
            EidsUtils.updateEidsIfUserInfoChanged(eidsRemoveByUserInfoChangedList)
        }

        if (userInfo.fpId.isEmpty()) {
            userInfo.fpId = DeviceUtils.getDeviceId(context)
        }
    }
}