package com.aotter.net.utils

import android.view.View
import android.view.ViewGroup
import androidx.core.view.forEach
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import com.aotter.net.dto.trek.response.TrekNativeAd
import com.aotter.net.extension.addObserverExt
import com.aotter.net.extension.getDetectPercent
import com.aotter.net.extension.removeObserverExt
import com.aotter.net.trek.ads.TrekMediaView
import com.aotter.net.trek.ads.vast.AotterPlayerView
import com.aotter.net.trek.ads.webview.AotterWebView
import com.aotter.net.trek.om.TrekOmCustomReferenceData
import com.aotter.net.trek.om.TrekOmFactory
import com.aotter.trek.impression.ImpressionListener
import com.aotter.trek.impression.ImpressionManager
import com.aotter.trek.impression.ImpressionProvider
import com.aotter.trek.impression.ImpressionRequest
import kotlinx.serialization.encodeToString

class ViewStateTracker(
    private val trekNativeAd: TrekNativeAd
) :
    View.OnAttachStateChangeListener,
    LifecycleEventObserver, ImpressionListener {

    private val TAG: String = ViewStateTracker::class.java.simpleName

    private var containerView: View? = null

    private var trekMediaView: TrekMediaView? = null

    private var impressionProvider: ImpressionProvider? = null

    private var lifecycle: Lifecycle? = null

    init {

        initOm()

    }

    override fun onViewAttachedToWindow(v: View) {
        resume()
    }

    override fun onViewDetachedFromWindow(v: View) {
        pause()
    }

    override fun onStateChanged(source: LifecycleOwner, event: Lifecycle.Event) {
        when (event) {
            Lifecycle.Event.ON_RESUME -> {
                resume()
            }

            Lifecycle.Event.ON_PAUSE -> {
                pause()
            }

            Lifecycle.Event.ON_DESTROY -> {
                destroy()
            }

            else -> Unit
        }
    }

    fun resume() {

        impressionProvider?.resume()

        trekMediaView?.resume()

    }

    fun pause() {

        impressionProvider?.pause()

        trekMediaView?.pause()

    }

    fun destroy() {

        destroyImpressionManager()

        lifecycle?.removeObserverExt(this)

        containerView?.removeOnAttachStateChangeListener(this@ViewStateTracker)

        trekNativeAd.trekOmFactory?.removeAllFriendlyObstructions()

        trekNativeAd.trekOmFactory?.stopAdSession()

        trekNativeAd.trekOmFactory = null

        trekMediaView?.destroy()

    }

    fun launchViewStateTracker(containerView: View, trekMediaView: TrekMediaView?) {

        this.trekMediaView = trekMediaView

        this.containerView = containerView.apply {

            this.addOnAttachStateChangeListener(this@ViewStateTracker)

            lifecycle = (this.context as? FragmentActivity)?.lifecycle

            lifecycle?.addObserverExt(this@ViewStateTracker)

            (containerView as? ViewGroup)?.let { viewGroup ->

                viewGroup.forEach {

                    addFriendlyObstruction(it)

                }

            }

            trekNativeAd.setNativeAdClickAction(containerView)

        }


        trekMediaView?.let { mediaview ->

            mediaview.setTrekMediaViewListener(object : TrekMediaView.TrekMediaViewListener {

                override fun onLoaded(view: View) {

                    launchImpression(mediaview)

                    when (view) {
                        is AotterWebView -> buildOmTracker(mediaview)
                        is AotterPlayerView -> {
                            mediaview.pause()
                            mediaview.resume()
                        }
                    }


                }

            })

            mediaview.load(containerView, trekNativeAd)

        } ?: kotlin.run {

            launchImpression(containerView)

            buildOmTracker(containerView)

        }

    }

    fun addFriendlyObstruction(view: View) {

        trekNativeAd.trekOmFactory?.addFriendlyObstruction(view)

        trekMediaView?.addFriendlyObstruction(view)

    }

    private fun buildOmTracker(view: View) {

        trekNativeAd.trekOmFactory?.registerAdView(view)

        trekNativeAd.trekOmFactory?.startAdSession()

        trekNativeAd.trekOmFactory?.loadedAdEvents()

    }

    private fun initOm() {

        trekNativeAd.trekOmFactory?.stopAdSession()

        trekNativeAd.trekOmFactory = null

        trekNativeAd.trekOmFactory = TrekOmFactory().apply {

            this.setContentUrl(trekNativeAd.contentUrl)

            this.setCustomReferenceData(
                TrekSdkSettingsUtils.json.encodeToString(
                    TrekOmCustomReferenceData(trekNativeAd.contentTitle)
                )
            )

            this.initNativeAdSession(trekNativeAd.omRes)

        }

    }

    private fun destroyImpressionManager() {

        impressionProvider?.destroy()

        impressionProvider = null

    }

    private fun launchImpression(registerView: View) {

        destroyImpressionManager()

        impressionProvider = ImpressionManager().with(registerView)

        val impressionRequest: ImpressionRequest =
            ImpressionRequest().setVisibleRangePercent(trekNativeAd.impressionSetting.impDetectPercent.getDetectPercent())
                .millisInFuture(trekNativeAd.impressionSetting.impRefreshMillis)

        val impressionListener = trekNativeAd.let {

            it.impressionListener

        } ?: kotlin.run {

            this

        }

        impressionProvider?.impressionRequest(impressionRequest)
            ?.impressionListener(impressionListener)?.viewStateTracker(this)?.apply()

    }

    override fun onImpression(view: View) {

        if (TrekSdkSettingsUtils.getUnitInstanceIdCache()[trekNativeAd.unitInstanceId] == trekNativeAd.unitInstanceId) return

        //這邊指專門處理 送 impression & om impression & 通知給開發者的 onimpression

        trekMediaView?.let { mediaview ->

            mediaview.notifyCatrunImpressionOccurred()

        } ?: kotlin.run {

            trekNativeAd.trekAdController?.getImpressionEvent(trekNativeAd.impressionUrl)

            trekNativeAd.thirdPartyImpUrls.forEach {
                trekNativeAd.trekAdController?.getThirdPartyImpressionEvent(it)
            }

        }

        trekNativeAd.trekOmFactory?.fireImpression()

        trekNativeAd.trekAdListener?.onAdImpression()

        TrekSdkSettingsUtils.setUnitInstanceIdCache(trekNativeAd.unitInstanceId)

    }

}