package com.aotter.trek.impression

import android.util.Log
import android.view.View
import android.view.ViewTreeObserver
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch


/**
 * 2020/06/08 created by Anthony Wu
 * 曝光提供者
 * 這裡會接收到 view 與 lifeCycle
 * 會依照職責配發任務下去給其他類，如下
 * ImpressionListener 負責曝光的回調監聽
 * ViewabilityListener 負責 可見區百分比 或是與 螢幕頂端的距離 的回調監聽
 * ViewVisibilityPercentageCalculator 負責計算可見百分比
 * ImpressionRequest 客製化的百分比參數與停留時間
 * ImpressionCountDownTimer 曝光計時器
 */
class ImpressionProvider(private val view: View) : ViewTreeObserver.OnDrawListener {

    private var impressionListener: ImpressionListener? = null

    private var viewStateTracker: ImpressionListener? = null

    private var impressionRequest = ImpressionRequest()

    private var impressionCountDownTimer: ImpressionCountDownTimer? = null

    private var viewVisibilityPercentageCalculator: ViewVisibilityPercentageCalculator? = null

    private val scope = CoroutineScope(Dispatchers.Main + SupervisorJob())

    fun impressionListener(impressionListener: ImpressionListener?): ImpressionProvider {
        this.impressionListener = impressionListener
        return this
    }

    fun viewStateTracker(viewStateTracker: ImpressionListener?): ImpressionProvider {
        this.viewStateTracker = viewStateTracker
        return this
    }

    fun impressionRequest(impressionRequest: ImpressionRequest): ImpressionProvider {
        this.impressionRequest = impressionRequest
        return this
    }


    fun apply() {

        destroy()

        view.viewTreeObserver.removeOnDrawListener(this)

        view.viewTreeObserver.addOnDrawListener(this)

        viewVisibilityPercentageCalculator = ViewVisibilityPercentageCalculator(view)

        impressionCountDownTimer =
            ImpressionCountDownTimer(
                view,
                impressionRequest,
                impressionListener,
                viewStateTracker
            )

        resume()

    }

    fun pause() {

        impressionCountDownTimer?.pause()

    }

    fun resume() {

        viewVisibilityPercentageCalculator?.reset()

        calculateViewPercentage()

    }

    fun destroy() {

        impressionCountDownTimer?.destroy()

    }

    private fun calculateViewPercentage() = scope.launch {

        viewVisibilityPercentageCalculator?.getVisibilityPercents()?.let { percent ->

            impressionCountDownTimer?.checkPercent(percent)

        }

        viewVisibilityPercentageCalculator?.getViewRatio()?.let {

            impressionCountDownTimer?.sendViewRatio(it)

        }

    }

    override fun onDraw() {
        calculateViewPercentage()
    }

}