package com.aotter.trek.impression

import android.graphics.Rect
import android.view.View
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock


/**
 * 2020/06/09 created by Anthony Wu
 * view可見度百分比計算者
 * 該類是針對view在螢幕上可見百分比的計算
 */
class ViewVisibilityPercentageCalculator(private val currentView: View) {

    private val location = IntArray(2)

    private var initLocalX = 0

    private var initLocalY = 0

    private var screenHeight = 0

    private var screenWidth = 0

    private var currentRatio = 0.0

    private var oldVisibilityPercents = -1

    private var initRunnable: kotlinx.coroutines.Runnable? = null

    private val visibilityPercentsMutex = Mutex()

    private val viewRatioMutex = Mutex()

    init {

        initRunnable = kotlinx.coroutines.Runnable {

            screenHeight = currentView.context.resources.displayMetrics.heightPixels

            screenWidth = currentView.context.resources.displayMetrics.widthPixels

            currentView.getLocationInWindow(location)

            initLocalX = location[0]

            initLocalY = location[1]

        }.apply {

            currentView.post(this)

        }

    }

    fun getScreenHeight() = screenHeight

    fun getScreenWidth() = screenWidth

    fun reset() {

        oldVisibilityPercents = -1

    }

    /**
     * 計算 view 的視野率
     */
    suspend fun getViewRatio(): Double? = viewRatioMutex.withLock {

        try {

            currentView.getLocationInWindow(location)

            val ratio = when {
                initLocalX == location[0] && initLocalY != location[1] -> 1.0 - (location[1].toDouble() / screenHeight.toDouble())
                initLocalX != location[0] && initLocalY == location[1] -> 1.0 - (location[0].toDouble() / screenWidth.toDouble())
                else -> 0.0
            }

            ratio.takeIf { it in 0.0..1.0 }?.let {
                val newRatio = "%.2f".format(currentRatio).toDouble()
                val oldRatio = "%.2f".format(it).toDouble()

                if (newRatio == oldRatio) null
                else {
                    currentRatio = it
                    newRatio
                }
            }

        } catch (e: Exception) {
            null
        }

    }

    /**
     * 計算可見百分比
     * 主要調用 view 的 getLocalVisibleRect 函數，取得 rect 去做計算
     */
    suspend fun getVisibilityPercents(): Int? = visibilityPercentsMutex.withLock {

        try {

            Rect(0, 0, screenWidth, screenHeight).run {

                var currentVisibilityPercents = -1

                if (!currentView.isShown) return null

                if (!isVisible(this)) return null

                val height = currentView.measuredHeight

                val width = currentView.measuredWidth

                when {

                    this.height() == height && this.width() == width -> currentVisibilityPercents =
                        (
                                100
                                )
                    viewIsPartiallyHiddenTop(this) -> currentVisibilityPercents =
                        ((height - this.top) * 100 / height)
                    viewIsPartiallyHiddenBottom(
                        this,
                        height
                    ) -> currentVisibilityPercents = (this.bottom * 100 / height)
                    viewIsPartiallyHiddenLeft(this) -> currentVisibilityPercents =
                        ((width - this.left) * 100 / width)
                    viewIsPartiallyHiddenRight(
                        this,
                        width
                    ) -> currentVisibilityPercents = (this.right * 100 / width)
                    else -> currentVisibilityPercents = 0

                }


                if ((oldVisibilityPercents == currentVisibilityPercents)
                    || currentVisibilityPercents <= -1
                    || currentVisibilityPercents > 100
                )
                    return null

                oldVisibilityPercents = currentVisibilityPercents

                return currentVisibilityPercents

            }


        } catch (e: Exception) {
            return null
        }

    }


    //view底部部分不可見
    private fun viewIsPartiallyHiddenBottom(
        rect: Rect,
        height: Int
    ): Boolean {
        return rect.bottom in 1 until height
    }

    //view頂部部分不可見
    private fun viewIsPartiallyHiddenTop(rect: Rect): Boolean {

        return rect.top > 0
    }


    //view右邊部分不可見
    private fun viewIsPartiallyHiddenRight(
        rect: Rect,
        width: Int
    ): Boolean {
        return rect.right in 1 until width
    }

    //view左邊部分不可見
    private fun viewIsPartiallyHiddenLeft(rect: Rect): Boolean {

        return rect.left > 0
    }

    private fun isVisible(actualPosition: Rect): Boolean {

        val locationOnScreen = currentView.getLocalVisibleRect(actualPosition)

        val screen = Rect(0, 0, screenWidth, screenHeight)

        return locationOnScreen && Rect.intersects(actualPosition, screen)

    }

}